<?php
/*
Plugin Name: AI-Content Writer
Plugin URI: https://www.en315.com/products/wordpress/
Description: AI 全自动文章创作与重写插件，集成标题生成、内容创作、图片生成、文章重写于一体。支持多 API 接入、自动生成模式、特色图片设置、SMTP 邮件通知等功能，提供完整的 AI 内容创作解决方案。
Version: 1.2.3
Update URI: false
License: GPL2
Author: En315
Author URI: https://www.en315.com
@package AIContentWriter
*/

if (!defined('ABSPATH')) exit;

// 禁用插件更新检查（针对新插件路径）
add_filter('site_transient_update_plugins', function($value) {
    if (isset($value) && is_object($value)) {
        unset($value->response['AI-Content Writer/ai-content-writer.php']);
    }
    return $value;
});

// 移除插件更新 nag
add_action('admin_init', function() {
    remove_action('admin_notices', 'update_nag', 3);
});

// 配置 SMTP（沿用原有选项键，方便迁移）
add_action('phpmailer_init', function($phpmailer) {
    $phpmailer->isSMTP();
    $phpmailer->Host = get_option('smtp_host');
    $phpmailer->SMTPAuth = true;
    $phpmailer->Username = get_option('smtp_username');
    $phpmailer->Password = get_option('smtp_password');
    $phpmailer->SMTPSecure = 'ssl';
    $phpmailer->Port = get_option('smtp_port');
    $phpmailer->CharSet = 'UTF-8';
});

class AI_Content_Writer {
    private $plugin_slug = 'ai-content-writer';
    private $card_api_base_url = 'https://card.en315.com/api';
    private $current_version = '1.2.3';
    private $version_check_url = 'https://www.en315.com/plugin-versions.json';
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_menu_page'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('enqueue_block_editor_assets', array($this, 'enqueue_editor_assets'));

        add_action('wp_ajax_generate_title', array($this, 'generate_title'));
        add_action('wp_ajax_delete_title', array($this, 'delete_title'));
        add_action('wp_ajax_delete_all_titles', array($this, 'delete_all_titles'));
        add_action('wp_ajax_update_auto_create', array($this, 'update_auto_create'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('wp_ajax_send_error_notification', array($this, 'send_error_notification'));
        add_action('wp_ajax_ai_smtp_test', array($this, 'ai_smtp_test'));
        // 文章与图片生成相关 AJAX
        add_action('wp_ajax_generate_moonshot_content', array($this, 'generate_moonshot_content'));
        add_action('wp_ajax_generate_image_prompt', array($this, 'generate_image_prompt'));
        add_action('wp_ajax_generate_moonshot_image', array($this, 'generate_moonshot_image'));
        
        // 卡密系统相关 AJAX
        add_action('wp_ajax_check_card_activation', array($this, 'check_card_activation'));
        add_action('wp_ajax_activate_card_frontend', array($this, 'activate_card_frontend'));
        
        // 版本更新检查（手动检查）
        add_action('wp_ajax_check_plugin_update_manual', array($this, 'check_plugin_update_manual'));
        
        // 文章重写相关 AJAX
        add_action('wp_ajax_extract_article_content', array($this, 'extract_article_content'));
        add_action('wp_ajax_add_article_to_list', array($this, 'add_article_to_list'));
        add_action('wp_ajax_remove_article_from_list', array($this, 'remove_article_from_list'));
        add_action('wp_ajax_get_article_list', array($this, 'get_article_list'));
        add_action('wp_ajax_rewrite_article', array($this, 'rewrite_article'));
        add_action('wp_ajax_create_rewrite_post', array($this, 'create_rewrite_post'));
        add_action('wp_ajax_get_post_content', array($this, 'get_post_content'));
        add_action('wp_ajax_get_unrewritten_articles', array($this, 'get_unrewritten_articles'));
        add_action('wp_ajax_update_article_status', array($this, 'update_article_status'));
        add_action('wp_ajax_send_rewrite_error_notification', array($this, 'send_rewrite_error_notification'));
        add_action('wp_ajax_check_post_status', array($this, 'check_post_status'));
        add_action('wp_ajax_clear_article_list', array($this, 'clear_article_list'));
    }

    /**
     * 文章重写功能：强制要求已激活
     */
    private function ensure_rewrite_activated_or_exit() {
        $activation_status = $this->check_remote_activation();
        if (!is_array($activation_status) || empty($activation_status['activated'])) {
            $msg = '文章重写功能需要激活插件';
            if (is_array($activation_status) && !empty($activation_status['message'])) {
                $msg .= '：' . $activation_status['message'];
            }
            wp_send_json_error($msg, 403);
            exit;
        }
    }
    
    /**
     * 手动检查插件更新（AJAX）
     */
    public function check_plugin_update_manual() {
        check_ajax_referer('ai_content_writer_check_update', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => '权限不足'));
            return;
        }
        
        // 清除缓存，强制检查
        delete_transient('ai_content_writer_update_check');
        $this->fetch_latest_version();
        set_transient('ai_content_writer_update_check', time(), DAY_IN_SECONDS);
        
        $latest_version = get_option('ai_content_writer_latest_version');
        $has_update = $latest_version && version_compare($this->current_version, $latest_version, '<');
        
        if ($has_update) {
            $update_url = get_option('ai_content_writer_update_url', 'https://www.en315.com/products/wordpress/');
            $update_message = get_option('ai_content_writer_update_message', '');
            wp_send_json_success(array(
                'has_update' => true,
                'current_version' => $this->current_version,
                'latest_version' => $latest_version,
                'update_url' => $update_url,
                'message' => $update_message
            ));
        } else {
            wp_send_json_success(array(
                'has_update' => false,
                'current_version' => $this->current_version,
                'message' => '当前已是最新版本'
            ));
        }
    }
    
    /**
     * 获取最新版本信息（从静态 JSON 文件）
     */
    private function fetch_latest_version() {
        $response = wp_remote_get($this->version_check_url, array(
            'timeout' => 10,
            'sslverify' => false
        ));
        
        if (!is_wp_error($response)) {
            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);
            
            if (isset($data['ai-content-writer']['latest_version'])) {
                $plugin_info = $data['ai-content-writer'];
                $latest_version = $plugin_info['latest_version'];
                
                if (version_compare($this->current_version, $latest_version, '<')) {
                    update_option('ai_content_writer_latest_version', $latest_version);
                    update_option('ai_content_writer_update_url', isset($plugin_info['download_url']) ? $plugin_info['download_url'] : 'https://www.en315.com/products/wordpress/');
                    update_option('ai_content_writer_update_message', isset($plugin_info['message']) ? $plugin_info['message'] : '');
                } else {
                    delete_option('ai_content_writer_latest_version');
                }
            }
        }
    }
    
    public function register_settings() {
        // 继续使用原来的选项组名，兼容旧数据
        register_setting('ai_title_generator_options', 'ai_title_prompt');
        register_setting('ai_title_generator_options', 'ai_user_prompt');
        register_setting('ai_title_generator_options', 'ai_auto_create_post');
        register_setting('ai_title_generator_options', 'ai_auto_generate_interval');
        register_setting('ai_title_generator_options', 'ai_page_refresh_interval');
        register_setting('ai_title_generator_options', 'ai_admin_notification_enabled');

        register_setting('ai_title_generator_options', 'ai_apis', array(
            'type' => 'array',
            'default' => array(
                'api1' => array(
                    'name'  => 'API1',
                    'key'   => '',
                    'url'   => '',
                    'model' => ''
                )
            )
        ));
        
        register_setting('ai_title_generator_options', 'ai_default_api', array(
            'type'    => 'string',
            'default' => 'deepseek'
        ));

        register_setting('ai_title_generator_options', 'smtp_host');
        register_setting('ai_title_generator_options', 'smtp_username');
        register_setting('ai_title_generator_options', 'smtp_password');
        register_setting('ai_title_generator_options', 'smtp_port');
        register_setting('ai_title_generator_options', 'notification_email');

        // 文章生成相关设置（来自 ai-content-generator）
        register_setting('ai_title_generator_options', 'moonshot_prompt_template');
        register_setting('ai_title_generator_options', 'chatfire_api_key');
        register_setting('ai_title_generator_options', 'generate_featured_image', 'intval');
        register_setting('ai_title_generator_options', 'image_insert_position', array(
            'type' => 'string',
            'default' => 'start'
        ));
        register_setting('ai_title_generator_options', 'image_api_url');
        register_setting('ai_title_generator_options', 'system_prompt_cn');
        register_setting('ai_title_generator_options', 'image_model');
        register_setting('ai_title_generator_options', 'image_apis', array(
            'type' => 'array',
            'default' => array(
                'img1' => array(
                    'name'  => '绘图通道',
                    'key'   => '',
                    'url'   => '',
                    'model' => '',
                    'aspectRatio' => 'auto',
                    'imageSize'   => '1K',
                )
            )
        ));
        register_setting('ai_title_generator_options', 'image_default_api', array(
            'type'    => 'string',
            'default' => 'img1'
        ));
        register_setting('ai_title_generator_options', 'image_prompt_template');
        register_setting('ai_title_generator_options', 'image_system_prompt');
        register_setting('ai_title_generator_options', 'image_hance_prompt');
        
        // 文章重写相关设置
        register_setting('ai_title_generator_options', 'rewrite_system_prompt');
        register_setting('ai_title_generator_options', 'rewrite_prompt_template');
        register_setting('ai_title_generator_options', 'rewrite_use_original_title', 'intval');
        register_setting('ai_title_generator_options', 'rewrite_title_prompt_template');
        register_setting('ai_title_generator_options', 'rewrite_title_system_prompt');

        add_settings_section(
            'ai_title_generator_section',
            '设置',
            null,
            'ai-content-writer-settings'
        );

        add_settings_field(
            'ai_title_prompt',
            '系统提示词',
            array($this, 'prompt_field_callback'),
            'ai-content-writer-settings',
            'ai_title_generator_section'
        );

        add_settings_field(
            'ai_user_prompt',
            '默认提示词',
            array($this, 'user_prompt_field_callback'),
            'ai-content-writer-settings',
            'ai_title_generator_section'
        );

        add_settings_field(
            'ai_auto_create_post',
            '自动生成文章',
            array($this, 'auto_create_post_callback'),
            'ai-content-writer-settings',
            'ai_title_generator_section'
        );

        add_settings_field(
            'ai_auto_generate_interval',
            '自动生成间隔',
            array($this, 'interval_field_callback'),
            'ai-content-writer-settings',
            'ai_title_generator_section'
        );

        add_settings_field(
            'ai_page_refresh_interval',
            '页面刷新间隔',
            array($this, 'page_refresh_interval_field_callback'),
            'ai-content-writer-settings',
            'ai_title_generator_section'
        );

        add_settings_field(
            'ai_admin_notification_enabled',
            '是否停止自动生成',
            array($this, 'admin_notification_field_callback'),
            'ai-content-writer-settings',
            'ai_title_generator_section'
        );

        add_settings_field(
            'ai_apis_settings',
            'API设置',
            array($this, 'apis_settings_callback'),
            'ai-content-writer-settings',
            'ai_title_generator_section'
        );

        add_settings_field(
            'smtp_settings',
            'SMTP设置',
            array($this, 'smtp_settings_callback'),
            'ai-content-writer-settings',
            'ai_title_generator_section'
        );

        // 文章与图片生成设置字段（统一到同一页面）
        add_settings_field(
            'content_image_settings',
            '文章与图片生成设置',
            array($this, 'content_image_settings_callback'),
            'ai-content-writer-settings',
            'ai_title_generator_section'
        );
    }

    public function prompt_field_callback() {
        $prompt = get_option('ai_title_prompt', '');
        ?>
        <textarea name="ai_title_prompt" rows="4" style="width: 100%; max-width: 800px; font-size: 16px; min-height: 100px;"><?php echo esc_textarea($prompt); ?></textarea>
        <p class="description">设置与 AI 对话时的系统提示词，这将影响生成标题和内容的风格。</p>
        <?php
    }

    public function user_prompt_field_callback() {
        $prompt = get_option('ai_user_prompt', '');
        ?>
        <textarea name="ai_user_prompt" rows="4" style="width: 100%; max-width: 800px; font-size: 16px; min-height: 100px;"><?php echo esc_textarea($prompt); ?></textarea>
        <p class="description">设置默认的提示词，在生成标题或内容时可以直接使用。</p>
        <?php
    }

    public function auto_create_post_callback() {
        $auto_create = get_option('ai_auto_create_post', '0');
        ?>
        <input type="hidden" name="ai_auto_create_post" id="ai_auto_create_post" value="<?php echo esc_attr($auto_create); ?>">
        <p class="description">在主页面使用“开始自动创建文章”和“停止自动创建文章”按钮来控制此功能。</p>
        <?php
    }

    public function interval_field_callback() {
        $interval = get_option('ai_auto_generate_interval', '5');
        ?>
        <input type="number" name="ai_auto_generate_interval" value="<?php echo esc_attr($interval); ?>" min="1" step="1" style="width: 100px;">
        <p class="description">设置自动生成标题的时间间隔（分钟），默认为 5 分钟。</p>
        <?php
    }

    public function page_refresh_interval_field_callback() {
        $refresh_interval = get_option('ai_page_refresh_interval', '10');
        ?>
        <input type="number" name="ai_page_refresh_interval" value="<?php echo esc_attr($refresh_interval); ?>" min="1" step="1" style="width: 100px;">
        <p class="description">设置每生成多少个标题后刷新页面，默认为 10 个，防止浏览器假死。</p>
        <?php
    }

    public function admin_notification_field_callback() {
        $admin_notification = get_option('ai_admin_notification_enabled', '0');
        ?>
        <input type="checkbox" name="ai_admin_notification_enabled" value="1" <?php checked($admin_notification, '1'); ?>>
        <p class="description">开启后，发送邮件通知的同时会停止自动创建文章功能，默认为关闭（仅发送邮件通知，不停止自动生成）。</p>
        <?php
    }

    public function add_menu_page() {
        $parent_slug = 'ai-content-writer';
        add_menu_page(
            'AI内容编写器',
            'AI内容编写器',
            'manage_options',
            $parent_slug,
            array($this, 'render_page'),
            'dashicons-edit'
        );
        add_submenu_page(
            $parent_slug,
            '设置',
            '设置',
            'manage_options',
            'ai-content-writer-settings',
            array($this, 'render_settings_page')
        );
    }

    public function render_settings_page() {
        // 预读取常用设置值，避免在模板中多次调用 get_option
        $ai_title_prompt      = get_option('ai_title_prompt', '');
        $ai_user_prompt       = get_option('ai_user_prompt', '');
        $ai_auto_create_post  = get_option('ai_auto_create_post', '0');
        $ai_auto_interval     = get_option('ai_auto_generate_interval', '5');
        $ai_page_refresh_int  = get_option('ai_page_refresh_interval', '10');
        $ai_admin_notify_stop = get_option('ai_admin_notification_enabled', '0');

        $smtp_host      = get_option('smtp_host');
        $smtp_username  = get_option('smtp_username');
        $smtp_password  = get_option('smtp_password');
        $smtp_port      = get_option('smtp_port');
        $notify_email   = get_option('notification_email');

        $system_prompt_cn = get_option('system_prompt_cn');
        $moonshot_tpl     = get_option('moonshot_prompt_template');

        $image_api_url       = get_option('image_api_url');
        $chatfire_api_key    = get_option('chatfire_api_key');
        $image_model         = get_option('image_model');
        $image_system_prompt = get_option('image_system_prompt');
        $image_prompt_tpl    = get_option('image_prompt_template');
        $generate_featured   = get_option('generate_featured_image', 0);
        $image_insert_pos    = get_option('image_insert_position', 'start');
        $image_hance_prompt  = get_option('image_hance_prompt');
        
        $rewrite_system_prompt = get_option('rewrite_system_prompt', '');
        $rewrite_prompt_template = get_option('rewrite_prompt_template', '');
        $rewrite_use_original_title = get_option('rewrite_use_original_title', 0);
        $rewrite_title_prompt_template = get_option('rewrite_title_prompt_template', '');
        $rewrite_title_system_prompt = get_option('rewrite_title_system_prompt', '');
        
        $latest_version = get_option('ai_content_writer_latest_version');
        $has_update = $latest_version && version_compare($this->current_version, $latest_version, '<');
        ?>
        <div class="wrap acw-settings-wrapper">
            <h1 class="acw-settings-title">
                AI-Content Writer 设置
                <span style="font-size: 14px; font-weight: normal; color: #666; margin-left: 10px;">
                    当前版本：<?php echo esc_html($this->current_version); ?>
                    <button type="button" id="acw-check-update-btn" class="button button-small" style="margin-left: 10px;">
                        <span class="acw-check-update-text">检查更新</span>
                    </button>
                    <span id="acw-update-status" style="margin-left: 10px;"></span>
                    <?php if ($has_update): ?>
                        <span id="acw-has-update" style="color: #d63638; margin-left: 10px;">有新版本 <?php echo esc_html($latest_version); ?> 可用</span>
                    <?php endif; ?>
                </span>
            </h1>
            <?php if ($has_update): ?>
                <div class="notice notice-info" style="margin: 15px 0;">
                    <p>
                        <strong>发现新版本！</strong> 当前版本：<?php echo esc_html($this->current_version); ?>，最新版本：<strong><?php echo esc_html($latest_version); ?></strong>
                        <?php 
                        $update_url = get_option('ai_content_writer_update_url', 'https://www.en315.com/products/wordpress/');
                        $update_message = get_option('ai_content_writer_update_message', '');
                        if ($update_message): 
                        ?>
                            <br><?php echo esc_html($update_message); ?>
                        <?php endif; ?>
                        <a href="<?php echo esc_url($update_url); ?>" target="_blank" class="button button-primary" style="margin-left: 10px;">立即更新</a>
                    </p>
                </div>
            <?php endif; ?>
            <form method="post" action="options.php">
                <?php settings_fields('ai_title_generator_options'); ?>

                <!-- Tab 导航 -->
                <nav class="acw-tab-nav">
                    <button type="button" class="acw-tab-btn active" data-tab="title">标题生成</button>
                    <button type="button" class="acw-tab-btn" data-tab="article">文章生成</button>
                    <button type="button" class="acw-tab-btn" data-tab="image">图片生成</button>
                    <button type="button" class="acw-tab-btn" data-tab="rewrite">文章重写</button>
                    <button type="button" class="acw-tab-btn" data-tab="ai-channel">AI通道</button>
                    <button type="button" class="acw-tab-btn" data-tab="global">全局设置</button>
                </nav>

                <!-- Tab 内容区域 -->
                <div class="acw-tab-content-wrapper">
                    <!-- 标题生成 Tab -->
                    <div class="acw-tab-panel active" id="tab-title">
                        <div class="acw-card">
                            <h2 class="acw-card-title">生成标题设置</h2>
                            <p class="acw-card-desc">配置系统提示词与默认提示词，定义标题生成的风格和约束。</p>
                            <div class="acw-card-body">
                                <div class="acw-field">
                                    <label for="ai_title_prompt">系统提示词</label>
                                    <textarea id="ai_title_prompt" name="ai_title_prompt" rows="4" class="acw-textarea"><?php echo esc_textarea($ai_title_prompt); ?></textarea>
                                    <p class="description">用于控制 AI 角色与语气，例如：你是一名专业的中文 SEO 标题编辑……</p>
                                </div>

                                <div class="acw-field">
                                    <label for="ai_user_prompt">默认提示词</label>
                                    <textarea id="ai_user_prompt" name="ai_user_prompt" rows="4" class="acw-textarea"><?php echo esc_textarea($ai_user_prompt); ?></textarea>
                                    <p class="description">生成标题时默认使用的提示词，可以在主界面根据需要临时修改。</p>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- 文章生成 Tab -->
                    <div class="acw-tab-panel" id="tab-article">
                        <div class="acw-card">
                            <h2 class="acw-card-title">生成文章设置</h2>
                            <p class="acw-card-desc">文章内容生成将使用 <strong>AI通道</strong> 中的通道配置（URL / API Key / 模型），这里仅配置提示词。</p>
                            <div class="acw-card-body">
                                <div class="acw-field">
                                    <label for="system_prompt_cn">文章系统提示词</label>
                                    <textarea id="system_prompt_cn" name="system_prompt_cn" rows="4" class="acw-textarea"><?php echo esc_textarea($system_prompt_cn); ?></textarea>
                                    <p class="description">控制文章整体风格、结构和语气（建议使用中文描述）。</p>
                                </div>
                                <div class="acw-field">
                                    <label for="moonshot_prompt_template">文章提示词模板</label>
                                    <textarea id="moonshot_prompt_template" name="moonshot_prompt_template" rows="4" class="acw-textarea"><?php echo esc_textarea($moonshot_tpl); ?></textarea>
                                    <p class="description">使用 <code>[标题]</code> 作为占位符，它会被实际文章标题替换。</p>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- 图片生成 Tab -->
                    <div class="acw-tab-panel" id="tab-image">
                        <div class="acw-card">
                            <h2 class="acw-card-title">生成图片设置</h2>
                            <p class="acw-card-desc">配置用于生成文章配图与特色图片的接口与模型。</p>
                            <div class="acw-card-body">
                                <div class="acw-field">
                                    <label for="image_system_prompt">图片系统提示词</label>
                                    <textarea id="image_system_prompt" name="image_system_prompt" rows="4" class="acw-textarea"><?php echo esc_textarea($image_system_prompt); ?></textarea>
                                    <p class="description">用于控制图片整体风格与画面特征，例如插画风格、镜头感等。</p>
                                </div>
                                <div class="acw-field">
                                    <label for="image_prompt_template">图片提示词模板</label>
                                    <textarea id="image_prompt_template" name="image_prompt_template" rows="4" class="acw-textarea"><?php echo esc_textarea($image_prompt_tpl); ?></textarea>
                                    <p class="description">使用 <code>{$title}</code> 和 <code>{$content}</code> 作为占位符，它们会被实际文章标题和内容替换。</p>
                                </div>
                                <div class="acw-field">
                                    <label>
                                        <input type="checkbox" name="generate_featured_image" value="1" <?php checked(1, $generate_featured); ?>>
                                        生成特色图片
                                    </label>
                                </div>
                                <div class="acw-field">
                                    <label for="image_insert_position">图片插入位置</label>
                                    <select id="image_insert_position" name="image_insert_position" class="acw-input">
                                        <option value="start" <?php selected($image_insert_pos, 'start'); ?>>文章开始</option>
                                        <option value="end" <?php selected($image_insert_pos, 'end'); ?>>文章结尾</option>
                                    </select>
                                    <p class="description">选择生成的内容图片插入到文章的位置。</p>
                                </div>
                                <div class="acw-field">
                                    <label for="image_hance_prompt">图片增强提示词</label>
                                    <input type="text" id="image_hance_prompt" name="image_hance_prompt" class="acw-input" value="<?php echo esc_attr($image_hance_prompt); ?>">
                                    <p class="description">在最终图片提示词末尾附加的增强描述，用于提升图片质量。</p>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- 文章重写 Tab -->
                    <div class="acw-tab-panel" id="tab-rewrite">
                        <div class="acw-card">
                            <h2 class="acw-card-title">文章重写设置</h2>
                            <p class="acw-card-desc">配置用于重写文章的系统提示词和提示词模板。</p>
                            <div class="acw-card-body">
                                <div class="acw-field">
                                    <label for="rewrite_system_prompt">重写文章系统提示词</label>
                                    <textarea id="rewrite_system_prompt" name="rewrite_system_prompt" rows="4" class="acw-textarea"><?php echo esc_textarea($rewrite_system_prompt); ?></textarea>
                                    <p class="description">控制文章重写的整体风格、结构和语气（建议使用中文描述）。</p>
                                </div>
                                <div class="acw-field">
                                    <label for="rewrite_prompt_template">重写文章提示词模板</label>
                                    <textarea id="rewrite_prompt_template" name="rewrite_prompt_template" rows="4" class="acw-textarea"><?php echo esc_textarea($rewrite_prompt_template); ?></textarea>
                                    <p class="description">使用 <code>[原文]</code> 和 <code>[标题]</code> 作为占位符，它们会被实际文章内容和标题替换。</p>
                                </div>
                                <div class="acw-field">
                                    <label>
                                        <input type="checkbox" name="rewrite_use_original_title" id="rewrite_use_original_title" value="1" <?php checked(1, $rewrite_use_original_title); ?>>
                                        使用原标题
                                    </label>
                                    <p class="description">启用后，重写文章时将使用原标题，不生成新标题。</p>
                                </div>
                                <div class="acw-field" id="rewrite_title_system_prompt_field" style="<?php echo $rewrite_use_original_title ? 'display: none;' : ''; ?>">
                                    <label for="rewrite_title_system_prompt">重写标题系统提示词</label>
                                    <textarea id="rewrite_title_system_prompt" name="rewrite_title_system_prompt" rows="4" class="acw-textarea"><?php echo esc_textarea($rewrite_title_system_prompt); ?></textarea>
                                    <p class="description">用于控制重写标题时 AI 的角色与语气，例如：你是一名专业的中文 SEO 标题编辑……仅在"使用原标题"开关为关时使用。</p>
                                </div>
                                <div class="acw-field" id="rewrite_title_template_field" style="<?php echo $rewrite_use_original_title ? 'display: none;' : ''; ?>">
                                    <label for="rewrite_title_prompt_template">重写标题提示词模板</label>
                                    <textarea id="rewrite_title_prompt_template" name="rewrite_title_prompt_template" rows="4" class="acw-textarea"><?php echo esc_textarea($rewrite_title_prompt_template); ?></textarea>
                                    <p class="description">使用 <code>[内容]</code> 和 <code>[标题]</code> 作为占位符，它们会被重写后的文章内容和原始标题替换。仅在"使用原标题"开关为关时使用。</p>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- AI通道 Tab -->
                    <div class="acw-tab-panel" id="tab-ai-channel">
                        <?php
                        $apis = get_option('ai_apis', array());
                        if (!is_array($apis)) {
                            $apis = array();
                        }
                        $deepseek_default_url = 'https://api.deepseek.com/v1/chat/completions';

                        // 默认通道：优先使用历史配置；若历史值不在当前通道列表中，则回落到 deepseek
                        $default_api_id = get_option('ai_default_api', 'deepseek');
                        if (!is_string($default_api_id)) {
                            $default_api_id = 'deepseek';
                        }

                        $get_api_field = function($api, $field) {
                            if (!is_array($api)) return '';
                            if (!isset($api[$field])) return '';
                            if (!is_string($api[$field]) && !is_numeric($api[$field])) return '';
                            return (string) $api[$field];
                        };

                        // 保障 DeepSeek 永远存在（不直接写库，仅用于渲染/提交）
                        $deepseek_api = isset($apis['deepseek']) && is_array($apis['deepseek']) ? $apis['deepseek'] : array();
                        $deepseek_api['name'] = 'DeepSeek';
                        $deepseek_url   = $get_api_field($deepseek_api, 'url');
                        $deepseek_key   = $get_api_field($deepseek_api, 'key');
                        $deepseek_model = $get_api_field($deepseek_api, 'model');
                        if (trim((string) $deepseek_url) === '') {
                            $deepseek_url = $deepseek_default_url;
                        }
                        if (trim((string) $deepseek_model) === '') {
                            $deepseek_model = 'deepseek-chat';
                        }

                        // 过滤掉明显的旧占位通道（api1 且完全为空），避免初次只想看到 DeepSeek 时出现噪音
                        $is_empty_placeholder = function($api_id, $api) use ($get_api_field) {
                            if ($api_id !== 'api1') return false;
                            $name  = $get_api_field($api, 'name');
                            $key   = $get_api_field($api, 'key');
                            $url   = $get_api_field($api, 'url');
                            $model = $get_api_field($api, 'model');
                            $name  = trim((string)$name);
                            $key   = trim((string)$key);
                            $url   = trim((string)$url);
                            $model = trim((string)$model);
                            return ($name === '' || $name === 'API1') && $key === '' && $url === '' && $model === '';
                        };

                        $valid_ids = array('deepseek' => true);
                        foreach ($apis as $api_id => $api) {
                            if ($api_id === 'deepseek') continue;
                            if (!is_array($api)) continue;
                            if ($is_empty_placeholder($api_id, $api)) continue;
                            $valid_ids[$api_id] = true;
                        }
                        if (!isset($valid_ids[$default_api_id])) {
                            $default_api_id = 'deepseek';
                        }

                        // ===== AI绘图通道（图片生成 API）：多通道 + 默认切换 =====
                        $image_apis = get_option('image_apis', array());
                        if (!is_array($image_apis)) {
                            $image_apis = array();
                        }
                        $image_default_api_id = get_option('image_default_api', 'img1');
                        if (!is_string($image_default_api_id)) {
                            $image_default_api_id = 'img1';
                        }

                        // 兼容旧单通道配置：image_api_url/chatfire_api_key/image_model
                        if (empty($image_apis)) {
                            $image_apis = array(
                                'img1' => array(
                                    'name'  => '绘图通道',
                                    'url'   => $image_api_url,
                                    'key'   => $chatfire_api_key,
                                    'model' => $image_model,
                                )
                            );
                            $image_default_api_id = 'img1';
                        }

                        // 固定绘图通道：nano-banana（不可删除）
                        if (!isset($image_apis['nano_banana']) || !is_array($image_apis['nano_banana'])) {
                            $image_apis['nano_banana'] = array(
                                'name'  => 'nano-banana',
                                'url'   => 'https://api.en315.com/v1/draw/nano-banana',
                                'key'   => '',
                                'model' => 'nano-banana-fast',
                                'aspectRatio' => 'auto',
                                'imageSize'   => '1K',
                            );
                        } else {
                            // 强制名称
                            $image_apis['nano_banana']['name'] = 'nano-banana';
                            if (!isset($image_apis['nano_banana']['model']) || trim((string)$image_apis['nano_banana']['model']) === '') {
                                $image_apis['nano_banana']['model'] = 'nano-banana-fast';
                            }
                            if (!isset($image_apis['nano_banana']['aspectRatio']) || trim((string)$image_apis['nano_banana']['aspectRatio']) === '') {
                                $image_apis['nano_banana']['aspectRatio'] = 'auto';
                            }
                            if (!isset($image_apis['nano_banana']['imageSize']) || trim((string)$image_apis['nano_banana']['imageSize']) === '') {
                                $image_apis['nano_banana']['imageSize'] = '1K';
                            }
                        }
                        // 默认通道不存在时，回落到第一个
                        if (!isset($image_apis[$image_default_api_id])) {
                            $image_api_keys = array_keys($image_apis);
                            $image_default_api_id = isset($image_api_keys[0]) ? $image_api_keys[0] : 'img1';
                        }
                        $default_image_api = isset($image_apis[$image_default_api_id]) && is_array($image_apis[$image_default_api_id]) ? $image_apis[$image_default_api_id] : array();
                        $legacy_image_api_url = $get_api_field($default_image_api, 'url');
                        $legacy_image_api_key = $get_api_field($default_image_api, 'key');
                        $legacy_image_model   = $get_api_field($default_image_api, 'model');
                        ?>

                        <div class="acw-card">
                            <h2 class="acw-card-title">AI对话通道</h2>
                            <p class="acw-card-desc">
                                选择默认通道并配置 URL / API Key / 模型。点击卡片进行配置。<br>
                                如使用中转api，请使用完整URL：<code>/v1/chat/completions</code>
                            </p>
                            <div class="acw-card-body">
                                <div class="acw-channel-grid">
                                    <?php
                                    // 1) DeepSeek 永远在第一个
                                    $deepseek_is_default    = ($default_api_id === 'deepseek');
                                    $deepseek_is_configured = (trim((string)$deepseek_key) !== '');
                                    ?>
                                    <button
                                        type="button"
                                        class="acw-channel-card<?php echo $deepseek_is_configured ? ' is-configured' : ' is-not-configured'; ?><?php echo $deepseek_is_default ? ' is-default' : ''; ?>"
                                        data-channel-id="deepseek"
                                        data-channel-name="DeepSeek"
                                        data-channel-url="<?php echo esc_attr($deepseek_url); ?>"
                                        data-channel-key="<?php echo esc_attr($deepseek_key); ?>"
                                        data-channel-model="<?php echo esc_attr($deepseek_model); ?>"
                                    >
                                        <span class="acw-channel-default-badge"<?php echo $deepseek_is_default ? '' : ' style="display:none"'; ?>>默认</span>
                                        <div class="acw-channel-name">DeepSeek</div>
                                        <div class="acw-channel-status">
                                            <span class="acw-channel-status-text<?php echo $deepseek_is_configured ? ' is-configured' : ' is-not-configured'; ?>">
                                                <?php echo $deepseek_is_configured ? '已配置' : '需配置'; ?>
                                            </span>
                                        </div>
                                    </button>

                                    <?php
                                    // 2) 其它已保存通道
                                    foreach ($apis as $api_id => $api) {
                                        if ($api_id === 'deepseek') continue;
                                        if (!is_array($api)) continue;
                                        if ($is_empty_placeholder($api_id, $api)) continue;

                                        $name  = trim($get_api_field($api, 'name'));
                                        $url   = $get_api_field($api, 'url');
                                        $key   = $get_api_field($api, 'key');
                                        $model = $get_api_field($api, 'model');
                                        if ($name === '') $name = '未命名通道';

                                        $is_default    = ($default_api_id === $api_id);
                                        $is_configured = (trim((string)$key) !== '');
                                        ?>
                                        <button
                                            type="button"
                                            class="acw-channel-card<?php echo $is_configured ? ' is-configured' : ' is-not-configured'; ?><?php echo $is_default ? ' is-default' : ''; ?>"
                                            data-channel-id="<?php echo esc_attr($api_id); ?>"
                                            data-channel-name="<?php echo esc_attr($name); ?>"
                                            data-channel-url="<?php echo esc_attr($url); ?>"
                                            data-channel-key="<?php echo esc_attr($key); ?>"
                                            data-channel-model="<?php echo esc_attr($model); ?>"
                                        >
                                            <span class="acw-channel-delete" role="button" tabindex="0" data-channel-id="<?php echo esc_attr($api_id); ?>" aria-label="删除通道">×</span>
                                            <span class="acw-channel-default-badge"<?php echo $is_default ? '' : ' style="display:none"'; ?>>默认</span>
                                            <div class="acw-channel-name"><?php echo esc_html($name); ?></div>
                                            <div class="acw-channel-status">
                                                <span class="acw-channel-status-text<?php echo $is_configured ? ' is-configured' : ' is-not-configured'; ?>">
                                                    <?php echo $is_configured ? '已配置' : '需配置'; ?>
                                                </span>
                                            </div>
                                        </button>
                                        <?php
                                    }
                                    ?>

                                    <!-- 3) 新增通道按钮（始终在最后） -->
                                    <button type="button" class="acw-channel-add-card" id="acw-channel-add-btn">
                                        <div class="acw-channel-add-plus">＋</div>
                                        <div class="acw-channel-add-text">新增通道</div>
                                    </button>
                                </div>

                                <!-- 绑定到原有 option：ai_apis / ai_default_api -->
                                <input type="hidden" id="acw-default-api" name="ai_default_api" value="<?php echo esc_attr($default_api_id); ?>">
                                <div id="acw-channel-hidden-fields">
                                    <!-- DeepSeek -->
                                    <input type="hidden" id="acw-channel-deepseek-name"  name="ai_apis[deepseek][name]"  value="DeepSeek">
                                    <input type="hidden" id="acw-channel-deepseek-url"   name="ai_apis[deepseek][url]"   value="<?php echo esc_attr($deepseek_url); ?>">
                                    <input type="hidden" id="acw-channel-deepseek-key"   name="ai_apis[deepseek][key]"   value="<?php echo esc_attr($deepseek_key); ?>">
                                    <input type="hidden" id="acw-channel-deepseek-model" name="ai_apis[deepseek][model]" value="<?php echo esc_attr($deepseek_model); ?>">

                                    <!-- 其它通道 -->
                                    <?php foreach ($apis as $api_id => $api): ?>
                                        <?php
                                        if ($api_id === 'deepseek') continue;
                                        if (!is_array($api)) continue;
                                        if ($is_empty_placeholder($api_id, $api)) continue;

                                        $name  = trim($get_api_field($api, 'name'));
                                        $url   = $get_api_field($api, 'url');
                                        $key   = $get_api_field($api, 'key');
                                        $model = $get_api_field($api, 'model');
                                        if ($name === '') $name = '未命名通道';
                                        ?>
                                        <input type="hidden" id="acw-channel-<?php echo esc_attr($api_id); ?>-name"  name="ai_apis[<?php echo esc_attr($api_id); ?>][name]"  value="<?php echo esc_attr($name); ?>">
                                        <input type="hidden" id="acw-channel-<?php echo esc_attr($api_id); ?>-url"   name="ai_apis[<?php echo esc_attr($api_id); ?>][url]"   value="<?php echo esc_attr($url); ?>">
                                        <input type="hidden" id="acw-channel-<?php echo esc_attr($api_id); ?>-key"   name="ai_apis[<?php echo esc_attr($api_id); ?>][key]"   value="<?php echo esc_attr($key); ?>">
                                        <input type="hidden" id="acw-channel-<?php echo esc_attr($api_id); ?>-model" name="ai_apis[<?php echo esc_attr($api_id); ?>][model]" value="<?php echo esc_attr($model); ?>">
                                    <?php endforeach; ?>
                                </div>

                                <!-- 配置弹窗（复用 settings-modern.css 的 .acw-modal 样式） -->
                                <div class="acw-modal" id="acw-channel-modal" style="display:none;">
                                    <div class="acw-modal-overlay" data-acw-modal-close="1"></div>
                                    <div class="acw-modal-content" role="dialog" aria-modal="true" aria-labelledby="acw-channel-modal-title">
                                        <div class="acw-modal-header">
                                            <h3 id="acw-channel-modal-title">配置通道</h3>
                                        </div>
                                        <div class="acw-modal-body">
                                            <input type="hidden" id="acw-channel-modal-id" value="">

                                            <div class="acw-field">
                                                <label for="acw-channel-modal-name">通道名称</label>
                                                <input type="text" id="acw-channel-modal-name" class="acw-input" placeholder="例如：豆包 / 自定义通道">
                                            </div>

                                            <div class="acw-field">
                                                <label for="acw-channel-modal-url">API URL</label>
                                                <input type="text" id="acw-channel-modal-url" class="acw-input" placeholder="例如：/v1/chat/completions">
                                            </div>

                                            <div class="acw-field">
                                                <label for="acw-channel-modal-key">API Key</label>
                                                <input type="text" id="acw-channel-modal-key" class="acw-input" placeholder="请输入 API Key">
                                            </div>

                                            <div class="acw-field">
                                                <label for="acw-channel-modal-model">模型</label>
                                                <input type="text" id="acw-channel-modal-model" class="acw-input" placeholder="例如：deepseek-chat / doubao-pro-256k">
                                                <select id="acw-channel-modal-model-select" class="acw-input" style="display:none;">
                                                    <option value="deepseek-chat">deepseek-chat</option>
                                                    <option value="deepseek-reasoner">deepseek-reasoner</option>
                                                </select>
                                            </div>

                                            <div class="acw-field">
                                                <label>
                                                    <input type="checkbox" id="acw-channel-modal-default" value="1">
                                                    设为默认通道
                                                </label>
                                            </div>
                                        </div>
                                        <div class="acw-modal-footer">
                                            <button type="button" class="button button-primary" id="acw-channel-modal-save">保存</button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                    <div class="acw-card">
                        <h2 class="acw-card-title">AI绘图通道</h2>
                        <p class="acw-card-desc">
                            配置生成图片所使用的 URL / API Key / 模型。点击卡片进行配置。<br>
                            如使用中转api，请使用完整URL：<code>/v1/chat/completions</code>
                        </p>
                        <div class="acw-card-body">
                            <div class="acw-channel-grid">
                                <?php
                                // 固定 nano-banana 永远排在第一个
                                $render_image_cards = array_merge(
                                    array('nano_banana' => $image_apis['nano_banana']),
                                    array_diff_key($image_apis, array('nano_banana' => true))
                                );
                                foreach ($render_image_cards as $img_id => $img_api):
                                ?>
                                    <?php
                                    if (!is_array($img_api)) continue;
                                    $img_name  = trim($get_api_field($img_api, 'name'));
                                    $img_url   = $get_api_field($img_api, 'url');
                                    $img_key   = $get_api_field($img_api, 'key');
                                    $img_model = $get_api_field($img_api, 'model');
                                    $img_aspect = $get_api_field($img_api, 'aspectRatio');
                                    $img_size   = $get_api_field($img_api, 'imageSize');
                                    if ($img_name === '') $img_name = '未命名绘图通道';

                                    $img_is_default    = ($image_default_api_id === $img_id);
                                    $img_is_configured = (trim((string)$img_key) !== '');
                                    $can_delete = ($img_id !== 'nano_banana') && (count($image_apis) > 1);
                                    if ($img_id === 'nano_banana') {
                                        if (trim((string)$img_aspect) === '') $img_aspect = 'auto';
                                        if (trim((string)$img_size) === '') $img_size = '1K';
                                    }
                                    ?>
                                    <button
                                        type="button"
                                        class="acw-image-channel-card<?php echo $img_is_configured ? ' is-configured' : ' is-not-configured'; ?><?php echo $img_is_default ? ' is-default' : ''; ?>"
                                        data-image-channel-id="<?php echo esc_attr($img_id); ?>"
                                        data-image-channel-name="<?php echo esc_attr($img_name); ?>"
                                        data-image-channel-url="<?php echo esc_attr($img_url); ?>"
                                        data-image-channel-key="<?php echo esc_attr($img_key); ?>"
                                        data-image-channel-model="<?php echo esc_attr($img_model); ?>"
                                        data-image-channel-aspect-ratio="<?php echo esc_attr($img_aspect); ?>"
                                        data-image-channel-image-size="<?php echo esc_attr($img_size); ?>"
                                    >
                                        <?php if ($can_delete): ?>
                                            <span class="acw-image-channel-delete" role="button" tabindex="0" data-image-channel-id="<?php echo esc_attr($img_id); ?>" aria-label="删除绘图通道">×</span>
                                        <?php endif; ?>
                                        <span class="acw-channel-default-badge"<?php echo $img_is_default ? '' : ' style="display:none"'; ?>>默认</span>
                                        <div class="acw-channel-name"><?php echo esc_html($img_name); ?></div>
                                        <div class="acw-channel-status">
                                            <span class="acw-channel-status-text<?php echo $img_is_configured ? ' is-configured' : ' is-not-configured'; ?>">
                                                <?php echo $img_is_configured ? '已配置' : '需配置'; ?>
                                            </span>
                                        </div>
                                    </button>
                                <?php endforeach; ?>

                                <button type="button" class="acw-channel-add-card" id="acw-image-channel-add-btn">
                                    <div class="acw-channel-add-plus">＋</div>
                                    <div class="acw-channel-add-text">新增绘图通道</div>
                                </button>
                            </div>

                            <!-- 绑定到新 option：image_apis / image_default_api -->
                            <input type="hidden" id="acw-image-default-api" name="image_default_api" value="<?php echo esc_attr($image_default_api_id); ?>">
                            <div id="acw-image-channel-hidden-fields">
                                <?php foreach ($render_image_cards as $img_id => $img_api): ?>
                                    <?php
                                    if (!is_array($img_api)) continue;
                                    $img_name  = trim($get_api_field($img_api, 'name'));
                                    $img_url   = $get_api_field($img_api, 'url');
                                    $img_key   = $get_api_field($img_api, 'key');
                                    $img_model = $get_api_field($img_api, 'model');
                                    $img_aspect = $get_api_field($img_api, 'aspectRatio');
                                    $img_size   = $get_api_field($img_api, 'imageSize');
                                    if ($img_name === '') $img_name = '未命名绘图通道';
                                    if ($img_id === 'nano_banana') {
                                        $img_name = 'nano-banana';
                                        if (trim((string)$img_model) === '') $img_model = 'nano-banana-fast';
                                        if (trim((string)$img_aspect) === '') $img_aspect = 'auto';
                                        if (trim((string)$img_size) === '') $img_size = '1K';
                                    }
                                    ?>
                                    <input type="hidden" id="acw-image-channel-<?php echo esc_attr($img_id); ?>-name"  name="image_apis[<?php echo esc_attr($img_id); ?>][name]"  value="<?php echo esc_attr($img_name); ?>">
                                    <input type="hidden" id="acw-image-channel-<?php echo esc_attr($img_id); ?>-url"   name="image_apis[<?php echo esc_attr($img_id); ?>][url]"   value="<?php echo esc_attr($img_url); ?>">
                                    <input type="hidden" id="acw-image-channel-<?php echo esc_attr($img_id); ?>-key"   name="image_apis[<?php echo esc_attr($img_id); ?>][key]"   value="<?php echo esc_attr($img_key); ?>">
                                    <input type="hidden" id="acw-image-channel-<?php echo esc_attr($img_id); ?>-model" name="image_apis[<?php echo esc_attr($img_id); ?>][model]" value="<?php echo esc_attr($img_model); ?>">
                                    <input type="hidden" id="acw-image-channel-<?php echo esc_attr($img_id); ?>-aspectRatio" name="image_apis[<?php echo esc_attr($img_id); ?>][aspectRatio]" value="<?php echo esc_attr($img_aspect); ?>">
                                    <input type="hidden" id="acw-image-channel-<?php echo esc_attr($img_id); ?>-imageSize" name="image_apis[<?php echo esc_attr($img_id); ?>][imageSize]" value="<?php echo esc_attr($img_size); ?>">
                                <?php endforeach; ?>
                            </div>

                            <!-- 兼容旧 option：始终同步默认绘图通道到旧字段 -->
                            <input type="hidden" id="acw-image-legacy-api-url" name="image_api_url" value="<?php echo esc_attr($legacy_image_api_url); ?>">
                            <input type="hidden" id="acw-image-legacy-api-key" name="chatfire_api_key" value="<?php echo esc_attr($legacy_image_api_key); ?>">
                            <input type="hidden" id="acw-image-legacy-model"   name="image_model" value="<?php echo esc_attr($legacy_image_model); ?>">

                            <!-- 绘图通道配置弹窗 -->
                            <div class="acw-modal" id="acw-image-channel-modal" style="display:none;">
                                <div class="acw-modal-overlay" data-acw-image-modal-close="1"></div>
                                <div class="acw-modal-content" role="dialog" aria-modal="true" aria-labelledby="acw-image-modal-title">
                                    <div class="acw-modal-header">
                                        <h3 id="acw-image-modal-title">配置绘图通道</h3>
                                    </div>
                                    <div class="acw-modal-body">
                                        <input type="hidden" id="acw-image-modal-id" value="">

                                        <div class="acw-field">
                                            <label for="acw-image-modal-name">通道名称</label>
                                            <input type="text" id="acw-image-modal-name" class="acw-input" placeholder="例如：绘图通道1 / 自定义绘图通道">
                                        </div>
                                        <div class="acw-field">
                                            <label for="acw-image-modal-url">API URL</label>
                                            <input type="text" id="acw-image-modal-url" class="acw-input" placeholder="使用官方完整URL：/v1/draw/nano-banana">
                                        </div>
                                        <div class="acw-field">
                                            <label for="acw-image-modal-key">API Key</label>
                                            <input type="text" id="acw-image-modal-key" class="acw-input" placeholder="请输入 API Key">
                                        </div>
                                        <div class="acw-field">
                                            <label for="acw-image-modal-model">模型</label>
                                            <input type="text" id="acw-image-modal-model" class="acw-input" placeholder="例如：nano-banana / dall-e-3 / flux 等">
                                            <select id="acw-image-modal-model-select" class="acw-input" style="display:none;">
                                                <option value="nano-banana-fast">nano-banana-fast</option>
                                                <option value="nano-banana">nano-banana</option>
                                                <option value="nano-banana-pro">nano-banana-pro</option>
                                                <option value="nano-banana-pro-vt">nano-banana-pro-vt</option>
                                                <option value="nano-banana-pro-cl">nano-banana-pro-cl</option>
                                                <option value="nano-banana-pro-vip">nano-banana-pro-vip</option>
                                                <option value="nano-banana-pro-4k-vip">nano-banana-pro-4k-vip</option>
                                            </select>
                                        </div>

                                        <div class="acw-field" id="acw-image-modal-aspect-field" style="display:none;">
                                            <label for="acw-image-modal-aspect">输出比例（aspectRatio）</label>
                                            <select id="acw-image-modal-aspect" class="acw-input">
                                                <option value="auto">auto</option>
                                                <option value="1:1">1:1</option>
                                                <option value="16:9">16:9</option>
                                                <option value="9:16">9:16</option>
                                                <option value="4:3">4:3</option>
                                                <option value="3:4">3:4</option>
                                                <option value="3:2">3:2</option>
                                                <option value="2:3">2:3</option>
                                                <option value="5:4">5:4</option>
                                                <option value="4:5">4:5</option>
                                                <option value="21:9">21:9</option>
                                            </select>
                                        </div>

                                        <div class="acw-field" id="acw-image-modal-size-field" style="display:none;">
                                            <label for="acw-image-modal-size">输出尺寸（imageSize）</label>
                                            <select id="acw-image-modal-size" class="acw-input">
                                                <option value="1K">1K</option>
                                                <option value="2K">2K</option>
                                                <option value="4K">4K</option>
                                            </select>
                                        </div>
                                        <div class="acw-field">
                                            <label>
                                                <input type="checkbox" id="acw-image-modal-default" value="1">
                                                设为默认绘图通道
                                            </label>
                                        </div>
                                    </div>
                                    <div class="acw-modal-footer">
                                        <button type="button" class="button button-primary" id="acw-image-modal-save">保存</button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    </div><!-- /#tab-ai-channel -->

                    <!-- 全局设置 Tab -->
                    <div class="acw-tab-panel" id="tab-global">
                        <div class="acw-card">
                            <h2 class="acw-card-title">全局设置</h2>
                            <p class="acw-card-desc">控制自动生成策略与系统通知方式。</p>
                            <div class="acw-card-body">
                                <div class="acw-field acw-field-inline">
                                    <label for="ai_auto_generate_interval">自动生成间隔</label>
                                    <div class="acw-inline-input">
                                        <input type="number" id="ai_auto_generate_interval" name="ai_auto_generate_interval" class="acw-input-small" value="<?php echo esc_attr($ai_auto_interval); ?>" min="1" step="1">
                                        <span class="acw-inline-suffix">分钟</span>
                                    </div>
                                    <p class="description">两次自动生成之间的时间间隔，建议 3–10 分钟。</p>
                                </div>

                                <div class="acw-field acw-field-inline">
                                    <label for="ai_page_refresh_interval">页面刷新间隔</label>
                                    <div class="acw-inline-input">
                                        <input type="number" id="ai_page_refresh_interval" name="ai_page_refresh_interval" class="acw-input-small" value="<?php echo esc_attr($ai_page_refresh_int); ?>" min="1" step="1">
                                        <span class="acw-inline-suffix">次生成后刷新</span>
                                    </div>
                                    <p class="description">为防止浏览器长时间运行导致卡顿，可定期刷新页面并恢复自动状态。</p>
                                </div>

                                <div class="acw-field">
                                    <label>
                                        <input type="checkbox" name="ai_admin_notification_enabled" value="1" <?php checked($ai_admin_notify_stop, '1'); ?>>
                                        出错时同时停止自动生成
                                    </label>
                                    <p class="description">开启后，发送错误通知邮件的同时会停止自动生成流程。</p>
                                </div>

                                <hr>

                                <div class="acw-field">
                                    <label for="smtp_host">SMTP 服务器</label>
                                    <input type="text" id="smtp_host" name="smtp_host" class="acw-input" value="<?php echo esc_attr($smtp_host); ?>">
                                    <p class="description">例如：<code>smtp.qq.com</code>、<code>smtp.gmail.com</code> 等。</p>
                                </div>

                                <div class="acw-field">
                                    <label for="smtp_username">SMTP 用户名</label>
                                    <input type="text" id="smtp_username" name="smtp_username" class="acw-input" value="<?php echo esc_attr($smtp_username); ?>">
                                </div>

                                <div class="acw-field">
                                    <label for="smtp_password">SMTP 密码/授权码</label>
                                    <input type="password" id="smtp_password" name="smtp_password" class="acw-input" value="<?php echo esc_attr($smtp_password); ?>">
                                </div>

                                <div class="acw-field acw-field-inline">
                                    <label for="smtp_port">SMTP 端口</label>
                                    <div class="acw-inline-input">
                                        <input type="number" id="smtp_port" name="smtp_port" class="acw-input-small" value="<?php echo esc_attr($smtp_port); ?>">
                                    </div>
                                </div>

                                <div class="acw-field">
                                    <label for="notification_email">通知邮箱</label>
                                    <input type="email" id="notification_email" name="notification_email" class="acw-input" value="<?php echo esc_attr($notify_email); ?>">
                                    <p class="description">用于接收 API 切换、系统错误、自动生成停止等通知。</p>
                                </div>

                                <div class="acw-field">
                                    <button type="button" id="acw-smtp-test" class="button button-secondary">发送测试邮件</button>
                                    <span id="acw-smtp-test-result" style="margin-left:8px;"></span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <?php submit_button('保存所有设置'); ?>
            </form>
        </div>

        <script>
        jQuery(document).ready(function($) {
            // Tab 切换功能
            $('.acw-tab-btn').on('click', function() {
                var targetTab = $(this).data('tab');
                
                // 移除所有活动状态
                $('.acw-tab-btn').removeClass('active');
                $('.acw-tab-panel').removeClass('active');
                
                // 添加当前活动状态
                $(this).addClass('active');
                $('#tab-' + targetTab).addClass('active');
                
                // 保存当前 tab 到 localStorage
                localStorage.setItem('acw-active-tab', targetTab);
            });
            
            // 恢复上次选择的 tab
            var savedTab = localStorage.getItem('acw-active-tab');
            if (savedTab) {
                $('.acw-tab-btn[data-tab="' + savedTab + '"]').trigger('click');
            }
            
            // 控制"重写标题提示词模板"和"重写标题系统提示词"字段的显示/隐藏
            function toggleRewriteTitleTemplate() {
                var useOriginalTitle = $('#rewrite_use_original_title').is(':checked');
                if (useOriginalTitle) {
                    $('#rewrite_title_template_field').hide();
                    $('#rewrite_title_system_prompt_field').hide();
                } else {
                    $('#rewrite_title_template_field').show();
                    $('#rewrite_title_system_prompt_field').show();
                }
            }
            
            // 初始化显示状态
            toggleRewriteTitleTemplate();
            
            // 监听开关变化
            $('#rewrite_use_original_title').on('change', function() {
                toggleRewriteTitleTemplate();
            });
            
            // SMTP 测试
            $('#acw-smtp-test').on('click', function() {
                var btn    = $(this);
                var result = $('#acw-smtp-test-result');
                btn.prop('disabled', true).text('发送中...');
                result.text('');
                $.post(ajaxurl, {
                    action: 'ai_smtp_test',
                    _ajax_nonce: '<?php echo wp_create_nonce('ai_content_writer'); ?>'
                }, function(res) {
                    btn.prop('disabled', false).text('发送测试邮件');
                    if (res.success) {
                        result.css('color', 'green').text('测试邮件发送成功，请检查邮箱');
                    } else {
                        result.css('color', 'red').text('发送失败：' + (res.data || '未知错误'));
                    }
                });
            });
            
            // 检查更新
            $('#acw-check-update-btn').on('click', function() {
                var btn = $(this);
                var statusEl = $('#acw-update-status');
                var hasUpdateEl = $('#acw-has-update');
                var textEl = btn.find('.acw-check-update-text');
                
                btn.prop('disabled', true);
                textEl.text('检查中...');
                statusEl.text('').css('color', '');
                
                $.post(ajaxurl, {
                    action: 'check_plugin_update_manual',
                    nonce: '<?php echo wp_create_nonce('ai_content_writer_check_update'); ?>'
                }, function(res) {
                    btn.prop('disabled', false);
                    textEl.text('检查更新');
                    
                    if (res.success) {
                        if (res.data.has_update) {
                            statusEl.css('color', '#d63638').text('有新版本 ' + res.data.latest_version + ' 可用');
                            if (hasUpdateEl.length === 0) {
                                $('<span id="acw-has-update" style="color: #d63638; margin-left: 10px;">有新版本 ' + res.data.latest_version + ' 可用</span>').insertAfter(statusEl);
                            } else {
                                hasUpdateEl.text('有新版本 ' + res.data.latest_version + ' 可用');
                            }
                            // 刷新页面以显示更新通知
                            setTimeout(function() {
                                location.reload();
                            }, 1500);
                        } else {
                            statusEl.css('color', 'green').text(res.data.message || '当前已是最新版本');
                            hasUpdateEl.remove();
                        }
                    } else {
                        statusEl.css('color', 'red').text('检查失败：' + (res.data.message || '未知错误'));
                    }
                }).fail(function() {
                    btn.prop('disabled', false);
                    textEl.text('检查更新');
                    statusEl.css('color', 'red').text('检查失败：网络错误');
                });
            });

            // ===== AI通道：卡片 + 弹窗配置 =====
            (function initAiChannelCards() {
                var $modal = $('#acw-channel-modal');
                if ($modal.length === 0) return;

                var $modalId     = $('#acw-channel-modal-id');
                var $modalTitle  = $('#acw-channel-modal-title');
                var $modalName   = $('#acw-channel-modal-name');
                var $modalUrl    = $('#acw-channel-modal-url');
                var $modalUrlDesc= $('#acw-channel-modal-url-desc');
                var $modalKey    = $('#acw-channel-modal-key');
                var $modalModel  = $('#acw-channel-modal-model');
                var $modalModelSelect = $('#acw-channel-modal-model-select');
                var $modalDefault= $('#acw-channel-modal-default');
                var $defaultApi  = $('#acw-default-api');
                var DEEPSEEK_DEFAULT_URL = 'https://api.deepseek.com/v1/chat/completions';
                var $hiddenWrap  = $('#acw-channel-hidden-fields');
                var $grid        = $('.acw-channel-grid');

                function openModalWithCard($card) {
                    var id    = $card.attr('data-channel-id') || '';
                    var label = $card.attr('data-channel-name') || $card.attr('data-channel-label') || '通道';
                    var url   = $card.attr('data-channel-url') || '';
                    var key   = $card.attr('data-channel-key') || '';
                    var model = $card.attr('data-channel-model') || '';
                    var currentDefault = ($defaultApi.val() || '').toString();

                    $modalId.val(id);
                    $modalTitle.text('配置通道 - ' + label);
                    $modalName.val(label);
                    if (id === 'deepseek') {
                        // DeepSeek：URL 可更改；为空时使用默认值
                        var urlVal = (url || '').toString().trim() || DEEPSEEK_DEFAULT_URL;
                        $modalUrl.val(urlVal).prop('readonly', false);
                        $modalUrlDesc.show();
                        $modalName.prop('readonly', true);
                    } else {
                        $modalUrl.val(url).prop('readonly', false);
                        $modalUrlDesc.hide();
                        $modalName.prop('readonly', false);
                    }
                    $modalKey.val(key);
                    if (id === 'deepseek') {
                        $modalModel.hide();
                        $modalModelSelect.show();
                        var selected = (model || '').toString().trim();
                        if (selected !== 'deepseek-chat' && selected !== 'deepseek-reasoner') {
                            selected = 'deepseek-chat';
                        }
                        $modalModelSelect.val(selected);
                    } else {
                        $modalModel.show();
                        $modalModelSelect.hide();
                        $modalModel.val(model);
                    }
                    $modalDefault.prop('checked', currentDefault === id);

                    $modal.show();
                }

                function openModalForAdd() {
                    $modalId.val('');
                    $modalTitle.text('新增通道');
                    $modalName.val('').prop('readonly', false);
                    $modalUrl.val('').prop('readonly', false);
                    $modalUrlDesc.hide();
                    $modalKey.val('');
                    $modalModel.val('');
                    $modalModel.show();
                    $modalModelSelect.hide();
                    $modalDefault.prop('checked', false);
                    $modal.show();
                }

                function closeModal() {
                    $modal.hide();
                }

                function refreshCardUI($card, isConfigured, isDefault) {
                    $card.toggleClass('is-configured', isConfigured);
                    $card.toggleClass('is-not-configured', !isConfigured);
                    $card.toggleClass('is-default', isDefault);

                    var $status = $card.find('.acw-channel-status-text');
                    $status.toggleClass('is-configured', isConfigured);
                    $status.toggleClass('is-not-configured', !isConfigured);
                    $status.text(isConfigured ? '已配置' : '需配置');

                    var $badge = $card.find('.acw-channel-default-badge');
                    if (isDefault) $badge.show();
                    else $badge.hide();
                }

                // 点击卡片打开弹窗
                $(document).on('click', '.acw-channel-card', function() {
                    openModalWithCard($(this));
                });

                // 删除通道（仅自定义通道）
                $(document).on('click', '.acw-channel-delete', function(e) {
                    e.preventDefault();
                    e.stopPropagation();
                    var id = ($(this).attr('data-channel-id') || '').toString();
                    if (!id || id === 'deepseek') return;
                    if (!confirm('确定要删除该通道吗？')) return;

                    // 移除卡片
                    $('.acw-channel-card[data-channel-id="' + id + '"]').remove();

                    // 移除隐藏字段
                    $('#acw-channel-' + id + '-name').remove();
                    $('#acw-channel-' + id + '-url').remove();
                    $('#acw-channel-' + id + '-key').remove();
                    $('#acw-channel-' + id + '-model').remove();

                    // 若删除的是默认通道，回退到 deepseek
                    if (($defaultApi.val() || '').toString() === id) {
                        $defaultApi.val('deepseek');
                        $('.acw-channel-card').each(function() {
                            var $c = $(this);
                            refreshCardUI($c, ($c.attr('data-channel-key') || '').toString().trim() !== '', $c.attr('data-channel-id') === 'deepseek');
                        });
                    }
                });

                // 允许键盘触发删除（Enter/Space）
                $(document).on('keydown', '.acw-channel-delete', function(e) {
                    if (e.key === 'Enter' || e.key === ' ') {
                        e.preventDefault();
                        $(this).trigger('click');
                    }
                });

                // 新增通道
                $(document).on('click', '#acw-channel-add-btn', function() {
                    openModalForAdd();
                });

                // 关闭弹窗（遮罩/取消按钮）
                // 用 document 委托 + 精确作用域到 #acw-channel-modal，避免委托顺序/冒泡导致“点取消不生效”
                $(document).on('click', '#acw-channel-modal [data-acw-modal-close="1"]', function(e) {
                    e.preventDefault();
                    closeModal();
                    return false;
                });

                // 保存配置
                $('#acw-channel-modal-save').on('click', function() {
                    var id = ($modalId.val() || '').toString().trim();
                    var isNew = !id;
                    var name = ($modalName.val() || '').toString().trim();
                    if (isNew) {
                        if (!name) {
                            alert('请输入通道名称');
                            return;
                        }
                        id = 'channel_' + Date.now();
                        $modalId.val(id);
                    } else if (id === 'deepseek') {
                        name = 'DeepSeek';
                    } else {
                        if (!name) name = '未命名通道';
                    }

                    var url   = ($modalUrl.val() || '').toString().trim();
                    var key   = ($modalKey.val() || '').toString().trim();
                    var model = ($modalModel.val() || '').toString().trim();
                    var setDefault = $modalDefault.is(':checked');

                    // DeepSeek：模型只能从下拉选择；URL 为空时回填默认值
                    if (id === 'deepseek') {
                        if (!url) url = DEEPSEEK_DEFAULT_URL;
                        model = ($modalModelSelect.val() || 'deepseek-chat').toString().trim();
                    }

                    // 生成/写回隐藏表单字段（保持 option 键不变）
                    function ensureHiddenInput(suffix, fieldName) {
                        var inputId = 'acw-channel-' + id + '-' + suffix;
                        var $input = $('#' + inputId);
                        if ($input.length) return $input;
                        $input = $('<input>', {
                            type: 'hidden',
                            id: inputId,
                            name: 'ai_apis[' + id + '][' + fieldName + ']'
                        });
                        $hiddenWrap.append($input);
                        return $input;
                    }
                    ensureHiddenInput('name', 'name').val(name);
                    ensureHiddenInput('url', 'url').val(url);
                    ensureHiddenInput('key', 'key').val(key);
                    ensureHiddenInput('model', 'model').val(model);

                    // 同步卡片 data-*，避免再次打开弹窗时取到旧值
                    var $card = $('.acw-channel-card[data-channel-id="' + id + '"]').first();
                    if ($card.length === 0) {
                        // 新增：插入卡片到“新增通道”按钮之前，按钮自然保持在最后
                        var $addBtn = $('#acw-channel-add-btn');
                        var cardHtml = `
                            <button type="button" class="acw-channel-card" data-channel-id="${id}">
                                <span class="acw-channel-delete" role="button" tabindex="0" data-channel-id="${id}" aria-label="删除通道">×</span>
                                <span class="acw-channel-default-badge" style="display:none">默认</span>
                                <div class="acw-channel-name"></div>
                                <div class="acw-channel-status">
                                    <span class="acw-channel-status-text"></span>
                                </div>
                            </button>
                        `;
                        $card = $(cardHtml);
                        $card.insertBefore($addBtn);
                    }
                    $card.attr('data-channel-name', name);
                    $card.attr('data-channel-url', url);
                    $card.attr('data-channel-key', key);
                    $card.attr('data-channel-model', model);
                    $card.find('.acw-channel-name').text(name);

                    // 更新默认
                    if (setDefault) {
                        $defaultApi.val(id);
                        $('.acw-channel-card').each(function() {
                            var $c = $(this);
                            refreshCardUI($c, ($c.attr('data-channel-key') || '').toString().trim() !== '', $c.attr('data-channel-id') === id);
                        });
                    } else {
                        refreshCardUI($card, key !== '', ($defaultApi.val() || '').toString() === id);
                    }

                    closeModal();
                });

                // ESC 关闭
                $(document).on('keydown', function(e) {
                    if (e.key === 'Escape' && $modal.is(':visible')) {
                        closeModal();
                    }
                });
            })();

            // ===== AI绘图通道：多通道卡片 + 弹窗配置（图片生成 API 配置）=====
            (function initAiImageChannelCards() {
                var $modal = $('#acw-image-channel-modal');
                if ($modal.length === 0) return;

                var $modalId     = $('#acw-image-modal-id');
                var $modalName   = $('#acw-image-modal-name');
                var $modalUrl    = $('#acw-image-modal-url');
                var $modalKey    = $('#acw-image-modal-key');
                var $modalModel  = $('#acw-image-modal-model');
                var $modalModelSelect = $('#acw-image-modal-model-select');
                var $aspectField = $('#acw-image-modal-aspect-field');
                var $aspectSelect= $('#acw-image-modal-aspect');
                var $sizeField   = $('#acw-image-modal-size-field');
                var $sizeSelect  = $('#acw-image-modal-size');
                var $modalDefault= $('#acw-image-modal-default');

                var $defaultApi  = $('#acw-image-default-api');
                var $hiddenWrap  = $('#acw-image-channel-hidden-fields');

                // legacy 同步（保持旧字段值与默认绘图通道一致，兼容老逻辑）
                var $legacyUrl   = $('#acw-image-legacy-api-url');
                var $legacyKey   = $('#acw-image-legacy-api-key');
                var $legacyModel = $('#acw-image-legacy-model');

                function closeModal() {
                    // 防止编辑 nano-banana 后名称保持只读影响“新增绘图通道”
                    $modalName.prop('readonly', false);
                    $modal.hide();
                }

                function openModalForAdd() {
                    $modalId.val('');
                    $modalName.val('').prop('readonly', false);
                    $modalUrl.val('');
                    $modalKey.val('');
                    $modalModel.val('');
                    $modalModel.show();
                    $modalModelSelect.hide();
                    $aspectField.hide();
                    $sizeField.hide();
                    $modalDefault.prop('checked', false);
                    $('#acw-image-modal-title').text('新增绘图通道');
                    $modal.show();
                }

                function openModalForEdit($card) {
                    var id    = ($card.attr('data-image-channel-id') || '').toString();
                    var name  = ($card.attr('data-image-channel-name') || '').toString();
                    var url   = ($card.attr('data-image-channel-url') || '').toString();
                    var key   = ($card.attr('data-image-channel-key') || '').toString();
                    var model = ($card.attr('data-image-channel-model') || '').toString();
                    var aspect = ($card.attr('data-image-channel-aspect-ratio') || '').toString();
                    var imgSize = ($card.attr('data-image-channel-image-size') || '').toString();
                    var currentDefault = ($defaultApi.val() || '').toString();

                    $modalId.val(id);
                    $modalName.val(name);
                    $modalUrl.val(url);
                    $modalKey.val(key);
                    if (id === 'nano_banana') {
                        $modalName.val('nano-banana').prop('readonly', true);
                        // 强制 nano-banana 接口路径存在
                        if (url && url.indexOf('/v1/draw/nano-banana') === -1) {
                            url = url.replace(/\/+$/, '') + '/v1/draw/nano-banana';
                            $modalUrl.val(url);
                        }
                        $modalModel.hide();
                        $modalModelSelect.show();
                        var selected = (model || '').toString().trim();
                        if (!selected) selected = 'nano-banana-fast';
                        $modalModelSelect.val(selected);
                        // nano-banana 才展示 aspect/imageSize
                        $aspectField.show();
                        $sizeField.show();
                        if (!aspect) aspect = 'auto';
                        if (!imgSize) imgSize = '1K';
                        $aspectSelect.val(aspect);
                        $sizeSelect.val(imgSize);
                        applyNanoImageSizeRules();
                    } else {
                        $modalName.prop('readonly', false);
                        $modalModel.show();
                        $modalModelSelect.hide();
                        $modalModel.val(model);
                        $aspectField.hide();
                        $sizeField.hide();
                    }
                    $modalDefault.prop('checked', currentDefault === id);
                    $('#acw-image-modal-title').text('配置绘图通道 - ' + (name || '绘图通道'));
                    $modal.show();
                }

                function ensureHiddenInput(id, suffix, fieldName) {
                    var inputId = 'acw-image-channel-' + id + '-' + suffix;
                    var $input = $('#' + inputId);
                    if ($input.length) return $input;
                    $input = $('<input>', {
                        type: 'hidden',
                        id: inputId,
                        name: 'image_apis[' + id + '][' + fieldName + ']'
                    });
                    $hiddenWrap.append($input);
                    return $input;
                }

                function applyNanoImageSizeRules() {
                    var model = ($modalModelSelect.val() || '').toString();
                    // 默认：都可选
                    $sizeSelect.find('option').prop('disabled', false);
                    if (model === 'nano-banana-pro-4k-vip') {
                        $sizeSelect.val('4K');
                        $sizeSelect.find('option[value="1K"], option[value="2K"]').prop('disabled', true);
                    } else if (model === 'nano-banana-pro-vip') {
                        // 只支持 1K/2K
                        if ($sizeSelect.val() === '4K') $sizeSelect.val('1K');
                        $sizeSelect.find('option[value="4K"]').prop('disabled', true);
                    }
                }

                function refreshCardUI($card, isConfigured, isDefault) {
                    $card.toggleClass('is-configured', isConfigured);
                    $card.toggleClass('is-not-configured', !isConfigured);
                    $card.toggleClass('is-default', isDefault);

                    var $status = $card.find('.acw-channel-status-text');
                    $status.toggleClass('is-configured', isConfigured);
                    $status.toggleClass('is-not-configured', !isConfigured);
                    $status.text(isConfigured ? '已配置' : '需配置');

                    var $badge = $card.find('.acw-channel-default-badge');
                    if (isDefault) $badge.show();
                    else $badge.hide();
                }

                function syncLegacyToDefault() {
                    var defId = ($defaultApi.val() || '').toString();
                    if (!defId) return;
                    var url   = ($('#acw-image-channel-' + defId + '-url').val() || '').toString();
                    var key   = ($('#acw-image-channel-' + defId + '-key').val() || '').toString();
                    var model = ($('#acw-image-channel-' + defId + '-model').val() || '').toString();
                    $legacyUrl.val(url);
                    $legacyKey.val(key);
                    $legacyModel.val(model);
                }

                function refreshAllCards() {
                    var defId = ($defaultApi.val() || '').toString();
                    $('.acw-image-channel-card').each(function() {
                        var $c = $(this);
                        var id = ($c.attr('data-image-channel-id') || '').toString();
                        var key = ($c.attr('data-image-channel-key') || '').toString().trim();
                        refreshCardUI($c, key !== '', id === defId);
                    });
                    syncLegacyToDefault();
                }

                // 点击卡片打开弹窗（排除新增按钮）
                $(document).on('click', '.acw-image-channel-card', function() {
                    openModalForEdit($(this));
                });

                // 新增绘图通道
                $(document).on('click', '#acw-image-channel-add-btn', function() {
                    openModalForAdd();
                });

                // 删除绘图通道
                $(document).on('click', '.acw-image-channel-delete', function(e) {
                    e.preventDefault();
                    e.stopPropagation();
                    var id = ($(this).attr('data-image-channel-id') || '').toString();
                    if (!id) return;
                    if (id === 'nano_banana') return;
                    // 至少保留一个
                    if ($('.acw-image-channel-card').length <= 1) return;
                    if (!confirm('确定要删除该绘图通道吗？')) return;

                    $('.acw-image-channel-card[data-image-channel-id="' + id + '"]').remove();
                    $('#acw-image-channel-' + id + '-name').remove();
                    $('#acw-image-channel-' + id + '-url').remove();
                    $('#acw-image-channel-' + id + '-key').remove();
                    $('#acw-image-channel-' + id + '-model').remove();

                    var defId = ($defaultApi.val() || '').toString();
                    if (defId === id) {
                        var $first = $('.acw-image-channel-card').first();
                        var newDef = ($first.attr('data-image-channel-id') || '').toString();
                        if (newDef) $defaultApi.val(newDef);
                    }

                    refreshAllCards();
                });

                // 键盘删除（Enter/Space）
                $(document).on('keydown', '.acw-image-channel-delete', function(e) {
                    if (e.key === 'Enter' || e.key === ' ') {
                        e.preventDefault();
                        $(this).trigger('click');
                    }
                });

                // 关闭弹窗（遮罩）
                $(document).on('click', '#acw-image-channel-modal [data-acw-image-modal-close="1"]', function(e) {
                    e.preventDefault();
                    closeModal();
                    return false;
                });

                // 保存（新增/编辑）
                $('#acw-image-modal-save').on('click', function() {
                    var id = ($modalId.val() || '').toString().trim();
                    var isNew = !id;
                    var name  = ($modalName.val() || '').toString().trim();
                    var url   = ($modalUrl.val() || '').toString().trim();
                    var key   = ($modalKey.val() || '').toString().trim();
                    var model = ($modalModel.val() || '').toString().trim();
                    var setDefault = $modalDefault.is(':checked');

                    if (!name) {
                        alert('请输入通道名称');
                        return;
                    }
                    if (isNew) {
                        id = 'img_' + Date.now();
                        $modalId.val(id);
                    }

                    // nano-banana 强约束：名称固定 + 模型下拉 + URL 确保为 /v1/draw/nano-banana
                    if (id === 'nano_banana') {
                        name = 'nano-banana';
                        model = ($modalModelSelect.val() || 'nano-banana-fast').toString().trim();
                        if (url && url.indexOf('/v1/draw/nano-banana') === -1) {
                            url = url.replace(/\/+$/, '') + '/v1/draw/nano-banana';
                        }
                        if (!url) {
                            url = 'https://api.en315.com/v1/draw/nano-banana';
                        }
                    }

                    var aspectRatio = '';
                    var imageSize = '';
                    if (id === 'nano_banana') {
                        aspectRatio = ($aspectSelect.val() || 'auto').toString();
                        imageSize = ($sizeSelect.val() || '1K').toString();
                        ensureHiddenInput(id, 'aspectRatio', 'aspectRatio').val(aspectRatio);
                        ensureHiddenInput(id, 'imageSize', 'imageSize').val(imageSize);
                    } else {
                        // 非 nano-banana：仍保存字段，但留空即可
                        ensureHiddenInput(id, 'aspectRatio', 'aspectRatio').val('');
                        ensureHiddenInput(id, 'imageSize', 'imageSize').val('');
                    }

                    ensureHiddenInput(id, 'name', 'name').val(name);
                    ensureHiddenInput(id, 'url', 'url').val(url);
                    ensureHiddenInput(id, 'key', 'key').val(key);
                    ensureHiddenInput(id, 'model', 'model').val(model);

                    var $card = $('.acw-image-channel-card[data-image-channel-id="' + id + '"]').first();
                    if ($card.length === 0) {
                        var $addBtn = $('#acw-image-channel-add-btn');
                        var cardHtml = `
                            <button type="button" class="acw-image-channel-card" data-image-channel-id="${id}">
                                <span class="acw-image-channel-delete" role="button" tabindex="0" data-image-channel-id="${id}" aria-label="删除绘图通道">×</span>
                                <span class="acw-channel-default-badge" style="display:none">默认</span>
                                <div class="acw-channel-name"></div>
                                <div class="acw-channel-status">
                                    <span class="acw-channel-status-text"></span>
                                </div>
                            </button>
                        `;
                        $card = $(cardHtml);
                        $card.insertBefore($addBtn);
                    }

                    $card.attr('data-image-channel-name', name);
                    $card.attr('data-image-channel-url', url);
                    $card.attr('data-image-channel-key', key);
                    $card.attr('data-image-channel-model', model);
                    if (id === 'nano_banana') {
                        $card.attr('data-image-channel-aspect-ratio', aspectRatio);
                        $card.attr('data-image-channel-image-size', imageSize);
                    }
                    $card.find('.acw-channel-name').text(name);

                    if (setDefault) {
                        $defaultApi.val(id);
                    }

                    refreshAllCards();
                    closeModal();
                });

                // ESC 关闭
                $(document).on('keydown', function(e) {
                    if (e.key === 'Escape' && $modal.is(':visible')) {
                        closeModal();
                    }
                });

                // nano-banana：模型变化时调整 imageSize 可选项
                $modalModelSelect.on('change', function() {
                    applyNanoImageSizeRules();
                });

                // 初始化同步 legacy
                refreshAllCards();
            })();
        });
        </script>
        <?php
    }

    public function render_page() {
        // 仅用于展示卡密状态，不在这里强制限制功能
        $activation_status = $this->check_remote_activation();
        
        $saved_titles   = get_option('ai_generated_titles', array());
        $default_prompt = get_option('ai_user_prompt', '');
        $auto_create    = get_option('ai_auto_create_post', '0');
        
        // 获取文章列表用于文章重写
        $posts = get_posts(array(
            'post_type' => 'post',
            'post_status' => 'publish',
            'numberposts' => 50,
            'orderby' => 'date',
            'order' => 'DESC'
        ));
        ?>
        <div class="wrap acw-main-wrapper">
            <h1>AI内容编写器</h1>
            
            <!-- 卡密激活状态区域（放在 tab 外层） -->
            <div id="card-activation-container"></div>
            
            <!-- 顶层 Tab 导航 -->
            <nav class="acw-main-tab-nav">
                <button type="button" class="acw-main-tab-btn active" data-main-tab="settings">文章创作</button>
                <button type="button" class="acw-main-tab-btn" data-main-tab="rewrite">文章重写</button>
            </nav>

            <!-- Tab 内容区域 -->
            <div class="acw-main-tab-content-wrapper">
                <!-- 文章创作 Tab -->
                <div class="acw-main-tab-panel active" id="main-tab-settings">
                    <?php if ($activation_status && !$activation_status['activated']): ?>
                    <script>
                    jQuery(document).ready(function($) {
                        // 只在需要时限制功能：这里不强制禁用按钮，实际限制在后端根据自动模式判断
                    });
                    </script>
                    <?php endif; ?>
                    
                    <div class="acw-rewrite-section">
                        <h2 style="margin: 0 0 20px 0;">标题列表</h2>
                        
                        <!-- 按钮组 -->
                        <div style="margin-bottom: 20px;">
                            <button id="generate-btn" class="button button-primary">生成标题</button>
                            <button id="start-auto-create" class="button button-secondary" style="margin-left: 10px;" <?php echo $auto_create === '1' ? 'style="display:none;"' : ''; ?>>
                                开始自动创建文章
                            </button>
                            <button id="stop-auto-create" class="button button-secondary" style="margin-left: 10px; display: none;" <?php echo $auto_create === '1' ? '' : 'style="display:none;"'; ?>>
                                停止自动创建文章
                            </button>
                            <button id="delete-all-titles" class="button" style="margin-left: 10px;">清空列表</button>
                        </div>
                        
                        <!-- 输入提示词 -->
                        <div class="form-group" style="margin-bottom: 20px;">
                            <label>输入提示词：</label>
                            <textarea id="prompt" rows="4" cols="50" style="width: 100%; max-width: 800px;"><?php echo esc_textarea($default_prompt); ?></textarea>
                        </div>
                        
                        <p class="description" style="margin-bottom: 20px;">请先激活插件，即可开始自动创建文章。<br> 
                        未激活时，只能手动生成标题和内容，不能自动创建文章。
                        </p>

                        <!-- 标题列表表格 -->
                        <div id="acw-title-list-container">
                            <table class="wp-list-table widefat fixed striped" id="acw-title-list-table">
                                <thead>
                                    <tr>
                                        <th width="5%">序号</th>
                                        <th>标题</th>
                                        <th width="15%">生成时间</th>
                                        <th width="15%">操作</th>
                                    </tr>
                                </thead>
                                <tbody id="acw-title-list-tbody">
                                    <?php if (empty($saved_titles)): ?>
                                    <tr id="acw-title-list-empty">
                                        <td colspan="4" style="text-align: center; padding: 20px; color: #999;">
                                            暂无标题，点击"生成标题"按钮生成
                                        </td>
                                    </tr>
                                    <?php else: ?>
                                    <?php foreach ($saved_titles as $index => $title_data): ?>
                                    <tr>
                                        <td><?php echo esc_html($index + 1); ?></td>
                                        <td><?php echo esc_html($title_data['title']); ?></td>
                                        <td><?php echo esc_html($title_data['date']); ?></td>
                                        <td>
                                            <?php if (isset($title_data['edit_link'])): ?>
                                            <a href="<?php echo esc_url($title_data['edit_link']); ?>" class="button button-small">编辑文章</a>
                                            <?php endif; ?>
                                            <button class="button button-small delete-title" data-index="<?php echo esc_attr($index); ?>">删除</button>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>

                <!-- 文章重写 Tab -->
                <div class="acw-main-tab-panel" id="main-tab-rewrite">
                    <div class="acw-rewrite-section">
                        <h2 style="margin: 0 0 20px 0;">文章列表</h2>
                        
                        <!-- 添加文章弹窗 -->
                        <div id="acw-add-article-modal" class="acw-modal" style="display: none;">
                            <div class="acw-modal-overlay"></div>
                            <div class="acw-modal-content">
                                <div class="acw-modal-header">
                                    <h3>添加文章</h3>
                                    <button type="button" class="acw-modal-close">&times;</button>
                                </div>
                                <div class="acw-modal-body">
                                    <div class="acw-field">
                                        <label for="acw-article-url">文章链接：</label>
                                        <div style="display: flex; gap: 10px; align-items: flex-start;">
                                            <input type="url" id="acw-article-url" class="regular-text" style="flex: 1;" placeholder="请输入文章链接，例如：https://example.com/article">
                                            <button type="button" id="acw-add-to-list-btn" class="button">添加至队列</button>
                                        </div>
                                        <p class="description">请输入要提取内容的文章链接，然后点击"添加至队列"</p>
                                    </div>
                                    <div id="acw-extract-status" style="margin-top: 15px;"></div>
                                </div>
                                <div class="acw-modal-footer">
                                    <button type="button" id="acw-extract-content-btn" class="button button-primary">查看内容</button>
                                    <button type="button" class="button acw-modal-cancel">取消</button>
                                </div>
                            </div>
                        </div>
                        
                        <!-- 打开弹窗按钮 -->
                        <div style="margin-bottom: 20px;">
                            <button type="button" id="acw-add-article-btn" class="button button-primary">
                                添加文章
                            </button>
                            <button id="start-auto-rewrite" class="button button-secondary" style="margin-left: 10px;">
                                开始自动重写
                            </button>
                            <button id="stop-auto-rewrite" class="button button-secondary" style="margin-left: 10px; display: none;">
                                停止自动重写
                            </button>
                            <button id="acw-clear-article-list-btn" class="button" style="margin-left: 10px;">
                                清空列表
                            </button>
                        </div>
                        
                        <!-- 文章链接列表 -->
                        <div id="acw-article-list-container">
                            <table class="wp-list-table widefat fixed striped" id="acw-article-list-table">
                                <thead>
                                    <tr>
                                        <th width="5%">序号</th>
                                        <th>标题</th>
                                        <th>链接</th>
                                        <th width="15%">添加时间</th>
                                        <th width="12%">状态</th>
                                        <th width="15%">操作</th>
                                    </tr>
                                </thead>
                                <tbody id="acw-article-list-tbody">
                                    <tr id="acw-article-list-empty">
                                        <td colspan="6" style="text-align: center; padding: 20px; color: #999;">
                                            暂无文章链接，点击"添加文章"按钮添加
                                        </td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            // 顶层 Tab 切换功能
            $('.acw-main-tab-btn').on('click', function() {
                var targetTab = $(this).data('main-tab');
                
                // 移除所有活动状态
                $('.acw-main-tab-btn').removeClass('active');
                $('.acw-main-tab-panel').removeClass('active');
                
                // 添加当前活动状态
                $(this).addClass('active');
                $('#main-tab-' + targetTab).addClass('active');
                
                // 保存当前 tab 到 localStorage
                localStorage.setItem('acw-main-active-tab', targetTab);

                // 未激活时：进入文章重写 tab 自动提示激活
                // 使用全局变量（由 server.js 提供）
                if (targetTab === 'rewrite' && window.acwActivationInfo && window.acwActivationInfo.activated === false) {
                    if (window.acwShowActivationForm) {
                        window.acwShowActivationForm(window.acwActivationInfo.message || '文章重写功能需要激活插件，请先激活卡密后再使用。');
                    }
                }
            });
            
            // 恢复上次选择的顶层 tab
            var savedMainTab = localStorage.getItem('acw-main-active-tab');
            if (savedMainTab) {
                $('.acw-main-tab-btn[data-main-tab="' + savedMainTab + '"]').trigger('click');
            }
        });
        </script>
        <?php
    }

    public function enqueue_scripts($hook) {
        // 文章编辑页：加载自动填充内容的脚本
        if ($hook === 'post-new.php' || $hook === 'post.php') {
            $post_editor_path = plugin_dir_path(__FILE__) . 'js/post-editor.js';
            $post_editor_ver  = file_exists($post_editor_path) ? filemtime($post_editor_path) : '1.0';
            wp_enqueue_script(
                'ai-content-writer-post-editor',
                plugins_url('js/post-editor.js', __FILE__),
                array('jquery', 'wp-blocks', 'wp-data', 'wp-editor', 'wp-block-editor'),
                $post_editor_ver,
                true
            );
            wp_localize_script('ai-content-writer-post-editor', 'aiTitleGenerator', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce'    => wp_create_nonce('ai_content_writer')
            ));
            wp_localize_script('ai-content-writer-post-editor', 'acwRewrite', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce'    => wp_create_nonce('acw_rewrite')
            ));
            return;
        }

        // 任何包含 ai-content-writer 的后台页面（主界面 + 设置页）都加载样式和管理脚本
        if (strpos($hook, 'ai-content-writer') === false) {
            return;
        }

        $admin_js_path  = plugin_dir_path(__FILE__) . 'js/admin.js';
        $rewrite_js_path = plugin_dir_path(__FILE__) . 'js/rewrite.js';
        $server_js_path  = plugin_dir_path(__FILE__) . 'js/server.js';
        $admin_js_ver   = file_exists($admin_js_path) ? filemtime($admin_js_path) : '1.0';
        $rewrite_js_ver = file_exists($rewrite_js_path) ? filemtime($rewrite_js_path) : '1.0';
        $server_js_ver  = file_exists($server_js_path) ? filemtime($server_js_path) : '1.0';

        wp_enqueue_script('ai-content-writer-admin', plugins_url('js/admin.js', __FILE__), array('jquery'), $admin_js_ver, true);
        wp_enqueue_script('ai-content-writer-rewrite', plugins_url('js/rewrite.js', __FILE__), array('jquery'), $rewrite_js_ver, true);
        wp_enqueue_script('ai-content-writer-server', plugins_url('js/server.js', __FILE__), array('jquery'), $server_js_ver, true);
        wp_enqueue_style('ai-content-writer-settings-modern', plugins_url('css/settings-modern.css', __FILE__));
        
        wp_localize_script('ai-content-writer-admin', 'aiTitleGenerator', array(
            'ajax_url'         => admin_url('admin-ajax.php'),
            'nonce'            => wp_create_nonce('ai_content_writer'),
            'interval'         => get_option('ai_auto_generate_interval', '5'),
            'refresh_interval' => get_option('ai_page_refresh_interval', '10')
        ));
        
        wp_localize_script('ai-content-writer-rewrite', 'acwRewrite', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce'    => wp_create_nonce('acw_rewrite')
        ));
        
        wp_localize_script('ai-content-writer-server', 'acwServer', array(
            'ajax_url'        => admin_url('admin-ajax.php'),
            'nonce'           => wp_create_nonce('ai_content_writer'),
            'autoCreateStatus' => get_option('ai_auto_create', '0') === '1'
        ));
    }

    /**
     * 区块编辑器侧边栏插件（内容 + 图片自动生成）
     */
    public function enqueue_editor_assets() {
        $editor_js_path = plugin_dir_path(__FILE__) . 'js/editor.js';
        $editor_js_ver  = file_exists($editor_js_path) ? filemtime($editor_js_path) : '1.0';
        wp_enqueue_script(
            'ai-content-writer-editor',
            plugins_url('js/editor.js', __FILE__),
            array('wp-plugins', 'wp-edit-post', 'wp-editor', 'wp-element', 'wp-components', 'wp-data', 'wp-compose'),
            $editor_js_ver,
            true
        );

        wp_localize_script('ai-content-writer-editor', 'moonshotSettings', array(
            'nonce'               => wp_create_nonce('moonshot_generate_content'),
            'apiUrl'              => admin_url('admin-ajax.php'),
            'debug'               => defined('WP_DEBUG') && WP_DEBUG,
            'generateFeaturedImage' => (bool) get_option('generate_featured_image', 0),
            'imageInsertPosition' => get_option('image_insert_position', 'start'),
        ));
    }

    /**
     * 文章与图片生成相关设置区域
     */
    public function content_image_settings_callback() {
        ?>
        <h3>生成文章设置</h3>
        <table class="form-table">
            <tr>
                <th scope="row">文章系统提示词</th>
                <td>
                    <textarea name="system_prompt_cn" rows="5" cols="50"><?php echo esc_textarea(get_option('system_prompt_cn')); ?></textarea>
                    <p class="description">生成文章时使用的系统提示词（角色塑造、内容风格、语气等）</p>
                </td>
            </tr>
            <tr>
                <th scope="row">文章提示词模板</th>
                <td>
                    <textarea name="moonshot_prompt_template" rows="5" cols="50"><?php echo esc_textarea(get_option('moonshot_prompt_template')); ?></textarea>
                    <p class="description">使用 [标题] 作为占位符，它会被实际文章标题替换</p>
                </td>
            </tr>
        </table>

        <h3>生成图片设置</h3>
        <table class="form-table">
            <tr>
                <th scope="row">图片系统提示词</th>
                <td>
                    <textarea name="image_system_prompt" rows="5" cols="50"><?php echo esc_textarea(get_option('image_system_prompt')); ?></textarea>
                    <p class="description">生成图片提示词（角色塑造、图片风格、画面描述等）</p>
                </td>
            </tr>
            <tr>
                <th scope="row">图片提示词模板</th>
                <td>
                    <textarea name="image_prompt_template" rows="5" cols="50"><?php echo esc_textarea(get_option('image_prompt_template')); ?></textarea>
                    <p class="description">使用 {$title} 和 {$content} 作为占位符，它们会被实际文章标题和内容替换</p>
                </td>
            </tr>
            <tr>
                <th scope="row">是否生成特色图片</th>
                <td>
                    <label>
                        <input type="checkbox" name="generate_featured_image" value="1" <?php checked(1, get_option('generate_featured_image', 0)); ?> />
                        生成特色图片
                    </label>
                </td>
            </tr>
            <tr>
                <th scope="row">图片增强提示词</th>
                <td>
                    <input type="text" name="image_hance_prompt" value="<?php echo esc_attr(get_option('image_hance_prompt')); ?>" class="regular-text" />
                    <p class="description">生成图片时附加的增强提示词，用于提升图片质量</p>
                </td>
            </tr>
        </table>
        <?php
    }

    public function generate_title() {
        check_ajax_referer('ai_content_writer');
        
        $prompt      = isset($_POST['prompt']) ? sanitize_textarea_field($_POST['prompt']) : '';
        $auto_create = get_option('ai_auto_create_post', '0');
        
        // 只在自动生成功能开启时校验卡密（手动点击生成标题无需卡密）
        if ($auto_create === '1' && !$this->can_use_plugin()) {
            wp_send_json_error('自动生成功能未激活或已过期，请先激活卡密后再使用自动生成功能');
            return;
        }
        
        // 保持原有的多 API 标题生成逻辑
        $apis = get_option('ai_apis', array());
        if (!is_array($apis)) {
            $apis = array();
        }
        $default_api_id = get_option('ai_default_api', 'deepseek');
        $deepseek_default_url = 'https://api.deepseek.com/v1/chat/completions';
        
        if (empty($apis)) {
            wp_send_json_error('未找到任何API配置');
            return;
        }
        
        // 默认 API 不存在时，回落到第一个可用 API（避免旧配置 api1 导致永远跳过）
        if (!is_string($default_api_id) || !isset($apis[$default_api_id])) {
            $api_keys = array_keys($apis);
            $default_api_id = isset($api_keys[0]) ? $api_keys[0] : 'deepseek';
        }

        $api_order = array($default_api_id);
        foreach ($apis as $api_id => $api) {
            if ($api_id !== $default_api_id) {
                $api_order[] = $api_id;
            }
        }

        $system_prompt  = get_option('ai_title_prompt');
        $error_messages = array();

        foreach ($api_order as $api_id) {
            if (!isset($apis[$api_id])) {
                continue;
            }

            $api = $apis[$api_id];

            // DeepSeek：URL 可更改（为空时使用默认值），模型可默认
            if ($api_id === 'deepseek') {
                $api['name']  = isset($api['name']) && $api['name'] ? $api['name'] : 'DeepSeek';
                if (!isset($api['url']) || trim((string)$api['url']) === '') {
                    $api['url'] = $deepseek_default_url;
                }
                $api['model'] = isset($api['model']) && trim((string)$api['model']) !== '' ? $api['model'] : 'deepseek-chat';
            }

            $api_name = isset($api['name']) && $api['name'] ? $api['name'] : $api_id;
            $missing = array();
            if (empty($api['key']))   $missing[] = 'API Key';
            if (empty($api['url']))   $missing[] = 'URL';
            if (empty($api['model'])) $missing[] = '模型';

            if (!empty($missing)) {
                $error_messages[] = "API {$api_name}: 配置不完整（缺少 " . implode('、', $missing) . "）";
                continue;
            }

            $response = wp_remote_post($api['url'], array(
                'headers' => array(
                    'Authorization' => 'Bearer ' . $api['key'],
                    'Content-Type'  => 'application/json',
                ),
                'body' => json_encode(array(
                    'model'    => $api['model'],
                    'messages' => array(
                        array(
                            'role'    => 'system',
                            'content' => $system_prompt
                        ),
                        array(
                            'role'    => 'user',
                            'content' => $prompt
                        )
                    ),
                    'temperature' => 1.2
                )),
                'timeout' => 300
            ));

            if (is_wp_error($response)) {
                $error_messages[] = "API {$api_name}: " . $response->get_error_message();
                continue;
            }

            $body = json_decode(wp_remote_retrieve_body($response), true);

            if (isset($body['error'])) {
                $error_message = $body['error']['message'] ?? '未知错误';
                $error_type    = $body['error']['type']    ?? '';
                $error_code    = $body['error']['code']    ?? '';

                $formatted_error = "API {$api['name']} 错误: ";
                if ($error_type) {
                    $formatted_error .= "[$error_type] ";
                }
                if ($error_code) {
                    $formatted_error .= "($error_code) ";
                }
                $formatted_error .= $error_message;

                // 尽量使用稳定的 API 名称
                $error_messages[] = str_replace("API {$api['name']} 错误:", "API {$api_name} 错误:", $formatted_error);
                continue;
            }

            if (!isset($body['choices'][0]['message']['content'])) {
                $error_messages[] = "API {$api_name}: 响应格式错误";
                continue;
            }

            $content = trim($body['choices'][0]['message']['content']);
            $titles  = array_filter(array_map('trim', explode("\n", $content)));

            $saved_titles = get_option('ai_generated_titles', array());

            $normalize = function($title) {
                $title = mb_strtolower($title, 'UTF-8');
                $title = preg_replace('/\s+/', ' ', $title);
                return trim($title);
            };

            $normalized_existing = array_map($normalize, array_column($saved_titles, 'title'));

            $new_titles = array();
            foreach ($titles as $title) {
                if (empty($title)) continue;

                $title = str_replace(array("\r"), '', $title);
                $title = preg_replace('/^\d+\.\s*/', '', $title);
                $title = str_replace(
                    array('！', '《', '》', '[', ']', ':', '：', '**'),
                    '',
                    $title
                );

                $title = trim($title, '""\""');
                $title = trim($title);

                $normalized_title = $normalize($title);

                if (in_array($normalized_title, $normalized_existing, true)) {
                    continue;
                }

                $title_data = array(
                    'title' => $title,
                    'date'  => current_time('Y-m-d H:i:s')
                );

                $new_titles[]          = $title_data;
                $normalized_existing[] = $normalized_title;
            }

            if (!empty($new_titles)) {
                if ($api_id !== $default_api_id) {
                    update_option('ai_default_api', $api_id);

                    if ($auto_create === '1') {
                        $failed_apis_details = array();
                        foreach ($error_messages as $error) {
                            if (preg_match('/API (.*?):(.*)/', $error, $matches)) {
                                $api_name    = trim($matches[1]);
                                $error_detail = trim($matches[2]);
                                $failed_apis_details[$api_name] = $error_detail;
                            }
                        }

                        $email_subject = 'API切换通知';
                        $email_message = '<div style="font-family: Arial, sans-serif; padding: 20px; max-width: 600px; margin: 0 auto; line-height: 1.6;">';
                        $email_message .= '<h2 style="color: #333; margin-bottom: 20px;">自动生成文章模式 - API切换成功通知</h2>';
                        $email_message .= '<p style="margin-bottom: 15px;"><strong>网站地址：</strong>' . get_site_url() . '</p>';
                        $email_message .= '<p style="margin-bottom: 15px;"><strong>成功生成标题的API是：</strong>' . $api['name'] . '</p>';
                        $email_message .= '<p style="margin-bottom: 15px;"><strong>生成标题失败的API是：</strong>' . implode(', ', array_keys($failed_apis_details)) . '</p>';
                        $email_message .= '<div style="background-color: #f8f8f8; padding: 15px; border-left: 4px solid #dc3545; margin-top: 15px;">';
                        $email_message .= '<strong>错误：</strong><br>' . implode('<br>', array_values($failed_apis_details));
                        $email_message .= '</div>';
                        $email_message .= '</div>';

                        $notification_email = get_option('notification_email');
                        $smtp_username      = get_option('smtp_username');
                        if (!empty($notification_email) && !empty($smtp_username)) {
                            $headers = array(
                                'Content-Type: text/html; charset=UTF-8',
                                'From: AI-Content Writer <'.$smtp_username.'>'
                            );
                            wp_mail(
                                array($notification_email),
                                $email_subject,
                                $email_message,
                                $headers
                            );
                        }
                    }
                }

                $all_titles = array_merge($saved_titles, $new_titles);
                update_option('ai_generated_titles', $all_titles);

                $first_title = $new_titles[0]['title'];

                wp_send_json_success(array(
                    'message'       => '成功使用 ' . $api['name'] . ' 生成' . count($new_titles) . '个标题',
                    'titles'        => $new_titles,
                    'first_title'   => $first_title,
                    'auto_create'   => $auto_create,
                    'new_post_url'  => admin_url('post-new.php'),
                    'api_switched'  => ($api_id !== $default_api_id),
                    'current_api'   => $api['name'],
                    'api_switch_log'=> array(
                        'default_api' => isset($apis[$default_api_id]['name']) ? $apis[$default_api_id]['name'] : '',
                        'used_api'    => $api['name'],
                        'failed_apis' => $error_messages
                    )
                ));
                return;
            } else {
                $error_messages[] = "API {$api['name']}: 所有生成的标题都已存在";
                continue;
            }
        }

        $error_message = "所有API都失败了：\n" . implode("\n", $error_messages);
        wp_send_json_error($error_message);
    }

    public function delete_title() {
        check_ajax_referer('ai_content_writer');
        
        $index        = intval($_POST['index']);
        $saved_titles = get_option('ai_generated_titles', array());
        
        if (isset($saved_titles[$index])) {
            array_splice($saved_titles, $index, 1);
            update_option('ai_generated_titles', $saved_titles);
            wp_send_json_success();
        } else {
            wp_send_json_error('标题不存在');
        }
    }

    public function update_auto_create() {
        check_ajax_referer('ai_content_writer');
        
        $value = sanitize_text_field($_POST['value']);
        update_option('ai_auto_create_post', $value);
        
        wp_send_json_success(array(
            'interval' => get_option('ai_auto_generate_interval', '5')
        ));
    }

    public function delete_all_titles() {
        check_ajax_referer('ai_content_writer');
        update_option('ai_generated_titles', array());
        wp_send_json_success();
    }

    public function send_error_notification() {
        check_ajax_referer('ai_content_writer');
        
        $notification_email        = get_option('notification_email');
        $smtp_username             = get_option('smtp_username');
        $admin_notification_enabled = get_option('ai_admin_notification_enabled', '0');
        $error_type                = isset($_POST['error_type']) ? sanitize_text_field($_POST['error_type']) : 'content';
        
        if (empty($notification_email) || empty($smtp_username)) {
            wp_send_json_error('请先在设置页面配置邮件通知信息');
        }
        
        $to      = array($notification_email);
        $subject = isset($_POST['subject']) ? sanitize_text_field($_POST['subject']) : '系统通知';
        $message = isset($_POST['message']) ? wp_kses_post($_POST['message']) : '系统已自动停止，请检查。';
        
        $html_message  = '<div style="font-family: Arial, sans-serif; padding: 20px; max-width: 600px; margin: 0 auto; line-height: 1.6;">';
        $html_message .= '<h2 style="color: #333; margin-bottom: 20px;">' . $subject . '</h2>';
        $html_message .= '<p style="margin-bottom: 15px;"><strong>网站地址：</strong>' . get_site_url() . '</p>';
        $html_message .= '<div style="background-color: #f8f8f8; padding: 15px; border-left: 4px solid #dc3545; margin-top: 15px;">';
        $html_message .= nl2br($message);
        $html_message .= '</div>';
        $html_message .= '</div>';
        
        $headers = array(
            'Content-Type: text/html; charset=UTF-8',
            'From: AI-Content Writer <'.$smtp_username.'>'
        );

        $sent = wp_mail($to, $subject, $html_message, $headers);
        
        if ($sent) {
            $response_data = array(
                'message'                    => '通知邮件已发送',
                'admin_notification_enabled' => $admin_notification_enabled
            );
            
            if ($error_type === 'title') {
                update_option('ai_auto_create_post', '0');
                $response_data['stop_auto_generate'] = true;
                $response_data['admin_notification'] = array(
                    'type'         => 'error',
                    'title'        => $subject,
                    'message'      => $message,
                    'auto_stopped' => true
                );
            } else {
                if ($admin_notification_enabled === '1') {
                    update_option('ai_auto_create_post', '0');
                    $response_data['stop_auto_generate'] = true;
                    $response_data['admin_notification'] = array(
                        'type'         => 'error',
                        'title'        => $subject,
                        'message'      => $message,
                        'auto_stopped' => true
                    );
                }
            }
            
            wp_send_json_success($response_data);
        } else {
            wp_send_json_error('邮件发送失败');
        }
    }

    public function apis_settings_callback() {
        $apis = get_option('ai_apis', array());
        if (!is_array($apis)) {
            $apis = array();
        }
        $default_api = get_option('ai_default_api', 'deepseek');
        ?>
        <div id="api-settings">
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th width="15%">API名称</th>
                        <th width="25%">API密钥</th>
                        <th width="25%">API地址</th>
                        <th width="15%">模型</th>
                        <th width="10%">默认</th>
                        <th width="10%">操作</th>
                    </tr>
                </thead>
                <tbody class="api-list">
                    <?php if (is_array($apis)) : foreach ($apis as $api_id => $api): ?>
                    <tr class="api-item" data-id="<?php echo esc_attr($api_id); ?>">
                        <td>
                            <input type="text" 
                                   name="ai_apis[<?php echo esc_attr($api_id); ?>][name]" 
                                   value="<?php echo esc_attr($api['name']); ?>" 
                                   placeholder="API名称"
                                   class="regular-text">
                        </td>
                        <td>
                            <input type="text" 
                                   name="ai_apis[<?php echo esc_attr($api_id); ?>][key]" 
                                   value="<?php echo esc_attr($api['key']); ?>" 
                                   placeholder="API密钥"
                                   class="regular-text">
                        </td>
                        <td>
                            <input type="text" 
                                   name="ai_apis[<?php echo esc_attr($api_id); ?>][url]" 
                                   value="<?php echo esc_attr($api['url']); ?>" 
                                   placeholder="API地址"
                                   class="regular-text">
                        </td>
                        <td>
                            <input type="text" 
                                   name="ai_apis[<?php echo esc_attr($api_id); ?>][model]" 
                                   value="<?php echo esc_attr($api['model']); ?>" 
                                   placeholder="模型"
                                   class="regular-text">
                        </td>
                        <td>
                            <input type="radio" 
                                   name="ai_default_api" 
                                   value="<?php echo esc_attr($api_id); ?>"
                                   <?php checked($default_api, $api_id); ?>>
                        </td>
                        <td>
                            <button type="button" class="button button-link-delete delete-api">删除</button>
                        </td>
                    </tr>
                    <?php endforeach; endif; ?>
                </tbody>
            </table>
            <p>
                <button type="button" class="button add-api">添加新API</button>
            </p>
        </div>
        <script>
        jQuery(document).ready(function($) {
            $('.add-api').on('click', function() {
                var apiId = 'api' + (Date.now());
                var template = `
                    <tr class="api-item" data-id="${apiId}">
                        <td>
                            <input type="text" 
                                   name="ai_apis[${apiId}][name]" 
                                   placeholder="API名称"
                                   class="regular-text">
                        </td>
                        <td>
                            <input type="text" 
                                   name="ai_apis[${apiId}][key]" 
                                   placeholder="API密钥"
                                   class="regular-text">
                        </td>
                        <td>
                            <input type="text" 
                                   name="ai_apis[${apiId}][url]" 
                                   placeholder="API地址"
                                   class="regular-text">
                        </td>
                        <td>
                            <input type="text" 
                                   name="ai_apis[${apiId}][model]" 
                                   placeholder="模型"
                                   class="regular-text">
                        </td>
                        <td>
                            <input type="radio" 
                                   name="ai_default_api" 
                                   value="${apiId}">
                        </td>
                        <td>
                            <button type="button" class="button button-link-delete delete-api">删除</button>
                        </td>
                    </tr>
                `;
                $('.api-list').append(template);
            });

            $(document).on('click', '.delete-api', function() {
                $(this).closest('.api-item').remove();
            });
        });
        </script>
        <style>
        .form-table th {
            padding: 10px 10px 10px 0;
            width: 120px;
            vertical-align: top;
        }
        .form-table td {
            padding: 10px 10px;
            vertical-align: top;
        }
        .form-table th label {
            white-space: nowrap;
        }
        .form-table tr {
            margin-bottom: 0;
        }
        .description {
            margin: 5px 0;
        }
        #wpbody-content {
            padding-bottom: 20px;
        }
        .wrap {
            margin: 10px 20px 0 0;
        }
        .wrap h1 {
            margin-bottom: 15px;
        }
        .form-table {
            margin-top: 0;
        }
        #api-settings {
            margin-top: 0;
        }
        #api-settings .wp-list-table {
            margin-top: 10px;
        }
        #api-settings .regular-text {
            width: 100%;
        }
        #api-settings td {
            vertical-align: middle;
            padding: 12px 10px;
        }
        #api-settings th {
            padding: 10px;
        }
        .add-api {
            margin: 10px 0;
        }
        .button-link-delete {
            color: #a00;
            text-decoration: none;
            border: none;
            background: transparent;
            padding: 0;
            cursor: pointer;
        }
        .button-link-delete:hover {
            color: #dc3232;
            background: transparent;
        }
        input[type="radio"] {
            margin: 0;
            vertical-align: middle;
        }
        .form-group {
            margin-bottom: 15px;
        }
        .form-group label {
            display: inline-block;
            margin-bottom: 5px;
            font-weight: 600;
            white-space: nowrap;
        }
        .form-group textarea {
            padding: 8px;
            line-height: 1.4;
            width: 100%;
            max-width: 800px;
            font-size: 16px;
            min-height: 100px;
        }
        .button-group {
            margin: 15px 0;
            display: flex;
            gap: 10px;
            align-items: center;
        }
        .titles-list {
            margin-top: 20px;
        }
        .titles-list h2 {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 10px;
        }
        textarea[name="ai_title_prompt"],
        textarea[name="ai_user_prompt"] {
            font-family: -apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,Oxygen-Sans,Ubuntu,Cantarell,"Helvetica Neue",sans-serif;
            padding: 10px;
            line-height: 1.6;
            border-radius: 4px;
            resize: vertical;
            margin-top: 5px;
        }
        .form-table td fieldset label {
            white-space: nowrap;
            margin: 0 !important;
            display: inline-block;
        }
        </style>
        <?php
    }

    public function smtp_settings_callback() {
        $smtp_host         = get_option('smtp_host');
        $smtp_username     = get_option('smtp_username');
        $smtp_password     = get_option('smtp_password');
        $smtp_port         = get_option('smtp_port');
        $notification_email = get_option('notification_email');
        ?>
        <p>
            <label>SMTP服务器：</label><br>
            <input type="text" name="smtp_host" value="<?php echo esc_attr($smtp_host); ?>" style="width: 100%; max-width: 600px;" required>
        </p>
        <p>
            <label>SMTP用户名：</label><br>
            <input type="text" name="smtp_username" value="<?php echo esc_attr($smtp_username); ?>" style="width: 100%; max-width: 600px;" required>
        </p>
        <p>
            <label>SMTP密码：</label><br>
            <input type="password" name="smtp_password" value="<?php echo esc_attr($smtp_password); ?>" style="width: 100%; max-width: 600px;" required>
        </p>
        <p>
            <label>SMTP端口：</label><br>
            <input type="number" name="smtp_port" value="<?php echo esc_attr($smtp_port); ?>" style="width: 100px;" required>
        </p>
        <p>
            <label>通知邮箱：</label><br>
            <input type="email" name="notification_email" value="<?php echo esc_attr($notification_email); ?>" style="width: 100%; max-width: 600px;" required>
        </p>
        <p>
            <button type="button" id="smtp-test-btn" class="button">发送测试邮件</button>
            <span id="smtp-test-result" style="margin-left:10px;"></span>
        </p>
        <script>
        jQuery(document).ready(function($) {
            $('#smtp-test-btn').on('click', function() {
                var btn    = $(this);
                var result = $('#smtp-test-result');
                btn.prop('disabled', true).text('发送中...');
                result.text('');
                $.post(ajaxurl, {
                    action: 'ai_smtp_test',
                    _ajax_nonce: '<?php echo wp_create_nonce('ai_content_writer'); ?>'
                }, function(res) {
                    btn.prop('disabled', false).text('发送测试邮件');
                    if (res.success) {
                        result.css('color', 'green').text('测试邮件发送成功，请检查邮箱');
                    } else {
                        result.css('color', 'red').text('发送失败：' + (res.data || '未知错误'));
                    }
                });
            });
        });
        </script>
        <?php
    }

    public function ai_smtp_test() {
        check_ajax_referer('ai_content_writer');
        $notification_email = get_option('notification_email');
        $smtp_username      = get_option('smtp_username');
        if (empty($notification_email) || empty($smtp_username)) {
            wp_send_json_error('请先填写完整SMTP和通知邮箱设置');
        }
        $subject = 'SMTP测试邮件 - AI-Content Writer';
        $message = '这是一封SMTP测试邮件，说明您的SMTP配置和通知邮箱设置正确。';
        $headers = array(
            'Content-Type: text/html; charset=UTF-8',
            'From: AI-Content Writer <'.$smtp_username.'>'
        );
        $sent = wp_mail($notification_email, $subject, $message, $headers);
        if ($sent) {
            wp_send_json_success();
        } else {
            wp_send_json_error('邮件发送失败，请检查SMTP配置');
        }
    }

    /**
     * 以下为原 ai-content-generator 的文章与图片生成逻辑
     */

    public function generate_moonshot_content() {
        header('Content-Type: application/json; charset=utf-8');
        
        try {
            if (!check_ajax_referer('moonshot_generate_content', 'nonce', false)) {
                wp_send_json_error('安全验证失败');
                return;
            }

            if (!current_user_can('edit_posts')) {
                wp_send_json_error('权限不足');
                return;
            }

            $title = isset($_POST['title']) ? sanitize_text_field($_POST['title']) : '';
            if (empty($title)) {
                wp_send_json_error('标题不能为空');
                return;
            }

            // 使用 AI通道（多API）配置，支持失败自动切换
            $apis = get_option('ai_apis', array());
            if (!is_array($apis)) {
                $apis = array();
            }
            if (empty($apis)) {
                wp_send_json_error('请先在 AI通道 中配置至少一个通道');
                return;
            }

            $default_api_id     = get_option('ai_default_api', 'deepseek');
            $deepseek_default_url = 'https://api.deepseek.com/v1/chat/completions';

            if (!is_string($default_api_id) || !isset($apis[$default_api_id])) {
                $api_keys = array_keys($apis);
                $default_api_id = isset($api_keys[0]) ? $api_keys[0] : 'deepseek';
            }

            $system_prompt   = get_option('system_prompt_cn');
            $prompt_template = get_option('moonshot_prompt_template');
            $prompt          = str_replace('[标题]', $title, $prompt_template);

            $api_order = array($default_api_id);
            foreach ($apis as $api_id => $api) {
                if ($api_id !== $default_api_id) {
                    $api_order[] = $api_id;
                }
            }

            $error_messages = array();

            foreach ($api_order as $api_id) {
                if (!isset($apis[$api_id]) || !is_array($apis[$api_id])) {
                    continue;
                }

                $api = $apis[$api_id];

                // DeepSeek：URL 可更改（为空时使用默认值），模型可默认
                if ($api_id === 'deepseek') {
                    $api['name']  = isset($api['name']) && $api['name'] ? $api['name'] : 'DeepSeek';
                    if (!isset($api['url']) || trim((string)$api['url']) === '') {
                        $api['url'] = $deepseek_default_url;
                    }
                    $api['model'] = isset($api['model']) && trim((string)$api['model']) !== '' ? $api['model'] : 'deepseek-chat';
                }

                $api_name = isset($api['name']) && $api['name'] ? $api['name'] : $api_id;
                $missing = array();
                if (empty($api['key']))   $missing[] = 'API Key';
                if (empty($api['url']))   $missing[] = 'URL';
                if (empty($api['model'])) $missing[] = '模型';

                if (!empty($missing)) {
                    $error_messages[] = "API {$api_name}: 配置不完整（缺少 " . implode('、', $missing) . "）";
                    continue;
                }

                $request_data = array(
                    'model'    => $api['model'],
                    'messages' => array(
                        array('role' => 'system', 'content' => $system_prompt),
                        array('role' => 'user',   'content' => $prompt)
                    ),
                    'temperature' => 0.7
                );

                $response = wp_remote_post($api['url'], array(
                    'headers' => array(
                        'Authorization' => 'Bearer ' . $api['key'],
                        'Content-Type'  => 'application/json'
                    ),
                    'body'      => json_encode($request_data, JSON_UNESCAPED_UNICODE),
                    'timeout'   => 120,
                    'sslverify' => false
                ));

                if (is_wp_error($response)) {
                    $error_messages[] = "API {$api_name}: " . $response->get_error_message();
                    continue;
                }

                $response_code = wp_remote_retrieve_response_code($response);
                $response_body = wp_remote_retrieve_body($response);

                if ($response_code !== 200) {
                    $error_message = json_decode($response_body, true);
                    $error_detail  = isset($error_message['error']['message']) ? $error_message['error']['message'] : $response_body;
                    $error_messages[] = "API {$api_name}: API响应错误: " . $error_detail;
                    continue;
                }

                $body = json_decode($response_body, true);
                if (!isset($body['choices'][0]['message']['content'])) {
                    $error_messages[] = "API {$api_name}: API响应格式错误";
                    continue;
                }

                $generated_content = $body['choices'][0]['message']['content'];

                // 生成成功：若发生切换，则更新默认 API（与标题生成一致）
                if ($api_id !== $default_api_id) {
                    update_option('ai_default_api', $api_id);
                }

                wp_send_json_success($generated_content);
                return;
            }

            wp_send_json_error("所有API都失败了：\n" . implode("\n", $error_messages));
            
        } catch (Exception $e) {
            wp_send_json_error('处理请求时发生错误: ' . $e->getMessage());
        }
    }

    public function generate_image_prompt() {
        header('Content-Type: application/json; charset=utf-8');
        
        try {
            if (!check_ajax_referer('moonshot_generate_content', 'nonce', false)) {
                wp_send_json_error('安全验证失败', 400);
                return;
            }

            $content = isset($_POST['content']) ? sanitize_text_field($_POST['content']) : '';
            $title   = isset($_POST['title']) ? sanitize_text_field($_POST['title']) : '';

            if (empty($content) || empty($title)) {
                wp_send_json_error('缺少必要参数', 400);
                return;
            }

            $system_prompt   = get_option('image_system_prompt');
            $prompt_template = get_option('image_prompt_template');
            $user_prompt     = str_replace(
                array('{$title}', '{$content}'),
                array($title, $content),
                $prompt_template
            );

            $request_data = array(
                // model 将从 AI通道配置中获取
                'model'    => '',
                'messages' => array(
                    array('role' => 'system', 'content' => $system_prompt),
                    array('role' => 'user',   'content' => $user_prompt)
                ),
                'temperature' => 0.3
            );

            // 使用 AI通道（多API）配置，支持失败自动切换
            $apis = get_option('ai_apis', array());
            if (!is_array($apis)) {
                $apis = array();
            }
            if (empty($apis)) {
                wp_send_json_error('请先在 AI通道 中配置至少一个通道');
                return;
            }

            $default_api_id     = get_option('ai_default_api', 'deepseek');
            $deepseek_default_url = 'https://api.deepseek.com/v1/chat/completions';

            if (!is_string($default_api_id) || !isset($apis[$default_api_id])) {
                $api_keys = array_keys($apis);
                $default_api_id = isset($api_keys[0]) ? $api_keys[0] : 'deepseek';
            }

            $api_order = array($default_api_id);
            foreach ($apis as $api_id => $api) {
                if ($api_id !== $default_api_id) {
                    $api_order[] = $api_id;
                }
            }

            $error_messages = array();
            $prompt = '';

            foreach ($api_order as $api_id) {
                if (!isset($apis[$api_id]) || !is_array($apis[$api_id])) {
                    continue;
                }

                $api = $apis[$api_id];
                if ($api_id === 'deepseek') {
                    $api['name']  = isset($api['name']) && $api['name'] ? $api['name'] : 'DeepSeek';
                    if (!isset($api['url']) || trim((string)$api['url']) === '') {
                        $api['url'] = $deepseek_default_url;
                    }
                    $api['model'] = isset($api['model']) && trim((string)$api['model']) !== '' ? $api['model'] : 'deepseek-chat';
                }

                $api_name = isset($api['name']) && $api['name'] ? $api['name'] : $api_id;
                $missing = array();
                if (empty($api['key']))   $missing[] = 'API Key';
                if (empty($api['url']))   $missing[] = 'URL';
                if (empty($api['model'])) $missing[] = '模型';
                if (!empty($missing)) {
                    $error_messages[] = "API {$api_name}: 配置不完整（缺少 " . implode('、', $missing) . "）";
                    continue;
                }

                $request_data['model'] = $api['model'];

                $response = wp_remote_post($api['url'], array(
                    'headers' => array(
                        'Authorization' => 'Bearer ' . $api['key'],
                        'Content-Type'  => 'application/json'
                    ),
                    'body'      => json_encode($request_data, JSON_UNESCAPED_UNICODE),
                    'timeout'   => 120,
                    'sslverify' => false
                ));

                if (is_wp_error($response)) {
                    $error_messages[] = "API {$api_name}: " . $response->get_error_message();
                    continue;
                }

                $response_code = wp_remote_retrieve_response_code($response);
                $response_body = wp_remote_retrieve_body($response);
                if ($response_code !== 200) {
                    $error_message = json_decode($response_body, true);
                    $error_detail  = isset($error_message['error']['message']) ? $error_message['error']['message'] : $response_body;
                    $error_messages[] = "API {$api_name}: API响应错误: " . $error_detail;
                    continue;
                }

                $body = json_decode($response_body, true);
                if (!isset($body['choices'][0]['message']['content'])) {
                    $error_messages[] = "API {$api_name}: API响应格式错误";
                    continue;
                }

                $prompt = $body['choices'][0]['message']['content'];

                if ($api_id !== $default_api_id) {
                    update_option('ai_default_api', $api_id);
                }

                break;
            }

            if ($prompt === '') {
                wp_send_json_error("所有API都失败了：\n" . implode("\n", $error_messages));
                return;
            }

            $original_prompt = $prompt;

            $max_length = 195;
            if (mb_strlen($prompt, 'UTF-8') > $max_length) {
                $prompt          = mb_substr($prompt, 0, $max_length, 'UTF-8');
                $original_prompt = $prompt;
            }

            $hance_prompt = get_option('image_hance_prompt');
            if (!empty($hance_prompt)) {
                $prompt .= $hance_prompt;
            }

            wp_send_json_success(array(
                'prompt'          => $prompt,
                'original_prompt' => $original_prompt
            ));
            
        } catch (Exception $e) {
            wp_send_json_error('处理请求时发生错误: ' . $e->getMessage());
        }
    }

    public function generate_moonshot_image() {
        header('Content-Type: application/json; charset=utf-8');
        
        try {
            if (!check_ajax_referer('moonshot_generate_content', 'nonce', false)) {
                wp_send_json_error('安全验证失败', 400);
                return;
            }

            if (!current_user_can('edit_posts')) {
                wp_send_json_error('权限不足', 400);
                return;
            }

            $prompt          = isset($_POST['prompt']) ? sanitize_text_field($_POST['prompt']) : '';
            $original_prompt = isset($_POST['original_prompt']) ? sanitize_text_field($_POST['original_prompt']) : '';
            $type            = isset($_POST['type']) ? sanitize_text_field($_POST['type']) : 'content';

            if (empty($prompt)) {
                wp_send_json_error('缺少必要参数', 400);
                return;
            }

            $size = $type === 'featured' ? '1024x768' : '1024x1024';

            // 使用 AI绘图通道（多通道）配置，支持失败自动切换
            $image_apis = get_option('image_apis', array());
            if (!is_array($image_apis)) {
                $image_apis = array();
            }
            $default_image_api_id = get_option('image_default_api', 'img1');
            if (!is_string($default_image_api_id)) {
                $default_image_api_id = 'img1';
            }

            // 兼容旧单通道配置
            if (empty($image_apis)) {
                $legacy_url   = get_option('image_api_url');
                $legacy_key   = get_option('chatfire_api_key');
                $legacy_model = get_option('image_model');
                if (empty($legacy_url) || empty($legacy_key) || empty($legacy_model)) {
                    wp_send_json_error('请先在 AI绘图通道 中配置绘图通道', 400);
                    return;
                }
                $image_apis = array(
                    'img1' => array(
                        'name'  => '绘图通道',
                        'url'   => $legacy_url,
                        'key'   => $legacy_key,
                        'model' => $legacy_model,
                    )
                );
                $default_image_api_id = 'img1';
            }

            if (!isset($image_apis[$default_image_api_id])) {
                $keys = array_keys($image_apis);
                $default_image_api_id = isset($keys[0]) ? $keys[0] : 'img1';
            }

            $api_order = array($default_image_api_id);
            foreach ($image_apis as $api_id => $api) {
                if ($api_id !== $default_image_api_id) {
                    $api_order[] = $api_id;
                }
            }

            $error_messages = array();

            foreach ($api_order as $api_id) {
                if (!isset($image_apis[$api_id]) || !is_array($image_apis[$api_id])) {
                    continue;
                }

                $api = $image_apis[$api_id];
                $api_name = isset($api['name']) && trim((string)$api['name']) !== '' ? (string)$api['name'] : $api_id;
                $api_key  = isset($api['key']) ? (string)$api['key'] : '';
                $api_url  = isset($api['url']) ? (string)$api['url'] : '';
                $model    = isset($api['model']) ? (string)$api['model'] : '';

                $missing = array();
                if (trim($api_key) === '') $missing[] = 'API Key';
                if (trim($api_url) === '') $missing[] = 'URL';
                if (trim($model) === '') $missing[] = '模型';
                if (!empty($missing)) {
                    $error_messages[] = "绘图通道 {$api_name}: 配置不完整（缺少 " . implode('、', $missing) . "）";
                    continue;
                }

                // === Nano Banana 专用协议 ===
                $is_nano_banana = ($api_id === 'nano_banana') || (strpos($api_url, '/v1/draw/nano-banana') !== false);
                if ($is_nano_banana) {
                    // 强约束：接口路径固定为 /v1/draw/nano-banana
                    if (strpos($api_url, '/v1/draw/nano-banana') === false) {
                        $api_url = rtrim($api_url, '/') . '/v1/draw/nano-banana';
                    }

                    $nano_model = trim($model) !== '' ? $model : 'nano-banana-fast';
                    $aspect_ratio = isset($api['aspectRatio']) && trim((string)$api['aspectRatio']) !== '' ? (string)$api['aspectRatio'] : 'auto';
                    $image_size   = isset($api['imageSize']) && trim((string)$api['imageSize']) !== '' ? (string)$api['imageSize'] : '1K';

                    $allowed_aspect = array('auto','1:1','16:9','9:16','4:3','3:4','3:2','2:3','5:4','4:5','21:9');
                    if (!in_array($aspect_ratio, $allowed_aspect, true)) {
                        $aspect_ratio = 'auto';
                    }
                    $allowed_size = array('1K','2K','4K');
                    if (!in_array($image_size, $allowed_size, true)) {
                        $image_size = '1K';
                    }
                    // 模型限制：vip(1k/2k)，4k-vip(仅4k)
                    if ($nano_model === 'nano-banana-pro-4k-vip') {
                        $image_size = '4K';
                    } elseif ($nano_model === 'nano-banana-pro-vip') {
                        if ($image_size === '4K') $image_size = '1K';
                    }
                    $nano_payload = array(
                        'model'       => $nano_model,
                        'prompt'      => $prompt,
                        'aspectRatio' => $aspect_ratio,
                        'imageSize'   => $image_size,
                        'urls'        => array(),
                        // 不使用 webhook，走流式；关闭进度直接返回最终结果（部分服务端会返回 JSON 或 SSE）
                        'shutProgress'=> false
                    );

                    $ch = curl_init();
                    curl_setopt_array($ch, array(
                        CURLOPT_URL            => $api_url,
                        CURLOPT_RETURNTRANSFER => true,
                        CURLOPT_POST           => true,
                        CURLOPT_POSTFIELDS     => json_encode($nano_payload, JSON_UNESCAPED_UNICODE),
                        CURLOPT_HTTPHEADER     => array(
                            'Authorization: Bearer ' . $api_key,
                            'Content-Type: application/json; charset=utf-8',
                            'Accept: application/json'
                        ),
                        CURLOPT_TIMEOUT        => 300,
                        CURLOPT_CONNECTTIMEOUT => 60,
                        CURLOPT_SSL_VERIFYPEER => false,
                    ));

                    $resp_raw   = curl_exec($ch);
                    $http_code  = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                    $curl_error = curl_error($ch);
                    curl_close($ch);

                    if ($curl_error) {
                        $error_messages[] = "绘图通道 {$api_name}: 请求失败: " . $curl_error;
                        continue;
                    }

                    if ($http_code !== 200) {
                        $error_messages[] = "绘图通道 {$api_name}: API响应错误: " . $resp_raw;
                        continue;
                    }

                    // 尝试从 JSON / 流式(SSE)中提取最终结果 JSON
                    $final = json_decode(trim((string)$resp_raw), true);
                    if (!is_array($final) || !isset($final['results'])) {
                        $final = null;
                        $lines = preg_split("/\r\n|\n|\r/", (string)$resp_raw);
                        foreach ($lines as $line) {
                            $line = trim($line);
                            if ($line === '' || $line === 'data:' || $line === '[DONE]') continue;
                            if (stripos($line, 'data:') === 0) {
                                $line = trim(substr($line, 5));
                            }
                            $j = json_decode($line, true);
                            if (is_array($j) && isset($j['results']) && is_array($j['results'])) {
                                $final = $j;
                            }
                        }
                    }

                    if (!is_array($final) || empty($final['results'][0]['url'])) {
                        $error_messages[] = "绘图通道 {$api_name}: 响应解析失败（未找到 results[0].url）";
                        continue;
                    }

                    $image_url = (string)$final['results'][0]['url'];

                    // 后续统一走下载入库流程（与下方逻辑一致）
                } else {
                $is_chat_api  = strpos($api_url, '/chat/completions') !== false;
                $image_model  = $model;
                $image_api_url = $api_url;

                if (strpos($image_model, 'stable-diffusion') !== false && $is_chat_api) {
                    $is_chat_api  = true;
                    $image_prompt = $prompt;
                } else {
                    $image_prompt = $prompt;
                }

                if ($is_chat_api) {
                    $request_data = array(
                        'model'    => $image_model,
                        'messages' => array(
                            array(
                                'role'    => 'user',
                                'content' => $image_prompt
                            )
                        ),
                        'max_tokens'  => 1000,
                        'temperature' => 0.7
                    );
                } elseif (strpos($image_model, 'dall-e') !== false || strpos($image_model, 'openai') !== false) {
                    $request_data = array(
                        'model'           => $image_model,
                        'prompt'          => $prompt,
                        'n'               => 1,
                        'size'            => $size,
                        'response_format' => 'url'
                    );
                } else {
                    $request_data = array(
                        'model'  => $image_model,
                        'prompt' => $prompt,
                        'n'      => 1,
                        'size'   => $size
                    );
                }

                $ch = curl_init();
                curl_setopt_array($ch, array(
                    CURLOPT_URL            => $image_api_url,
                    CURLOPT_RETURNTRANSFER => true,
                    CURLOPT_POST           => true,
                    CURLOPT_POSTFIELDS     => json_encode($request_data, JSON_UNESCAPED_UNICODE),
                    CURLOPT_HTTPHEADER     => array(
                        'Authorization: Bearer ' . $api_key,
                        'Content-Type: application/json; charset=utf-8',
                        'Accept: application/json'
                    ),
                    CURLOPT_TIMEOUT        => 120,
                    CURLOPT_CONNECTTIMEOUT => 60,
                    CURLOPT_SSL_VERIFYPEER => false,
                ));

                $response   = curl_exec($ch);
                $http_code  = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                $curl_error = curl_error($ch);
                curl_close($ch);

                if ($curl_error) {
                    $error_messages[] = "绘图通道 {$api_name}: 请求失败: " . $curl_error;
                    continue;
                }

                if ($http_code !== 200) {
                    $error_message = json_decode($response, true);
                    $error_detail  = isset($error_message['error']['message']) ? $error_message['error']['message'] : $response;
                    $error_messages[] = "绘图通道 {$api_name}: API响应错误: " . $error_detail;
                    continue;
                }

                $body      = json_decode($response, true);
                $image_url = null;

                if ($is_chat_api) {
                    if (isset($body['choices'][0]['message']['content'])) {
                        $content = trim($body['choices'][0]['message']['content']);

                        if (preg_match('/^https?:\/\//i', $content)) {
                            $image_url = $content;
                        } elseif (preg_match('/!\[.*?\]\((https?:\/\/[^\s\)]+)\)/i', $content, $matches)) {
                            $image_url = $matches[1];
                        } elseif (preg_match('/(https?:\/\/[^\s<>"]+\.(jpg|jpeg|png|gif|webp)(\?[^\s<>"]*)?)/i', $content, $matches)) {
                            $image_url = $matches[1];
                        } elseif (preg_match('/(https?:\/\/[^\s<>"]+)/i', $content, $matches)) {
                            $image_url = $matches[1];
                        }
                    }
                } else {
                    if (isset($body['data'][0]['url'])) {
                        $image_url = $body['data'][0]['url'];
                    } elseif (isset($body['data']['url'])) {
                        $image_url = $body['data']['url'];
                    } elseif (isset($body['url'])) {
                        $image_url = $body['url'];
                    } elseif (isset($body['image_url'])) {
                        $image_url = $body['image_url'];
                    }
                }

                if (!$image_url) {
                    $error_messages[] = "绘图通道 {$api_name}: API响应格式错误，无法找到图片URL";
                    continue;
                }
                }

                // 兼容 nano-banana 分支中未设置 $image_url 的情况（这里确保有值）
                if (empty($image_url)) {
                    $error_messages[] = "绘图通道 {$api_name}: 未获取到图片URL";
                    continue;
                }

                $download = wp_remote_get($image_url);
                if (is_wp_error($download)) {
                    $error_messages[] = "绘图通道 {$api_name}: 下载图片失败: " . $download->get_error_message();
                    continue;
                }

                $image_data = wp_remote_retrieve_body($download);
                if (empty($image_data)) {
                    $error_messages[] = "绘图通道 {$api_name}: 获取到的图片数据为空";
                    continue;
                }

                $upload_dir = wp_upload_dir();
                $filename   = 'generated-image-' . time() . '.png';
                $file_path  = $upload_dir['path'] . '/' . $filename;

                if (!file_put_contents($file_path, $image_data)) {
                    $error_messages[] = "绘图通道 {$api_name}: 保存图片失败";
                    continue;
                }

                $attachment = array(
                    'post_mime_type' => 'image/png',
                    'post_title'     => $filename,
                    'post_content'   => '',
                    'post_status'    => 'inherit'
                );

                $attach_id = wp_insert_attachment($attachment, $file_path);
                if (is_wp_error($attach_id)) {
                    $error_messages[] = "绘图通道 {$api_name}: 创建附件失败";
                    continue;
                }

                require_once(ABSPATH . 'wp-admin/includes/image.php');
                $attach_data = wp_generate_attachment_metadata($attach_id, $file_path);
                wp_update_attachment_metadata($attach_id, $attach_data);

                // 成功：若发生切换，则更新默认绘图通道，并同步旧配置
                if ($api_id !== $default_image_api_id) {
                    update_option('image_default_api', $api_id);
                }
                update_option('image_api_url', $image_api_url);
                update_option('chatfire_api_key', $api_key);
                update_option('image_model', $image_model);

                $result = array(
                    'url'             => $upload_dir['url'] . '/' . $filename,
                    'id'              => $attach_id,
                    'original_prompt' => $original_prompt
                );

                wp_send_json_success($result);
                return;
            }

            wp_send_json_error("所有绘图通道都失败了：\n" . implode("\n", $error_messages), 500);
            
        } catch (Exception $e) {
            wp_send_json_error('处理请求时发生错误: ' . $e->getMessage(), 500);
        }
    }
    
    /**
     * 只在需要使用自动生成功能时检查是否可以使用插件
     */
    private function can_use_plugin() {
        $cache_key     = 'ai_content_writer_activation_cache';
        $cached_result = get_transient($cache_key);
        
        if ($cached_result !== false) {
            return $cached_result;
        }
        
        $result = $this->verify_remote_activation();
        
        set_transient($cache_key, $result, 300);
        
        return $result;
    }
    
    private function verify_remote_activation() {
        $domain = home_url();
        
        $response = wp_remote_post($this->card_api_base_url . '/verify.php', array(
            'body'    => json_encode(array(
                'domain'            => $domain,
                'plugin_identifier' => $this->plugin_slug
            )),
            'headers' => array(
                'Content-Type' => 'application/json'
            ),
            'timeout' => 10
        ));
        
        if (is_wp_error($response)) {
            // 网络错误时允许使用，避免影响正常功能
            return true;
        }
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        return isset($data['success']) && $data['success'] && isset($data['activated']) && $data['activated'];
    }
    
    private function check_remote_activation() {
        // 短缓存：避免同一时刻多个 AJAX/页面加载并发触发 verify.php，导致卡密端产生重复日志
        $cache_key = 'ai_content_writer_activation_status_cache';
        $cached = get_transient($cache_key);
        if ($cached !== false && is_array($cached) && array_key_exists('activated', $cached)) {
            return $cached;
        }

        $domain = home_url();
        
        $response = wp_remote_post($this->card_api_base_url . '/verify.php', array(
            'body'    => json_encode(array(
                'domain'            => $domain,
                'plugin_identifier' => $this->plugin_slug
            )),
            'headers' => array(
                'Content-Type' => 'application/json'
            ),
            'timeout' => 5
        ));
        
        if (is_wp_error($response)) {
            $result = array(
                'activated' => false,
                'message'   => '无法连接到授权服务器'
            );
            set_transient($cache_key, $result, 60);
            return $result;
        }
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        if (isset($data['success']) && $data['success']) {
            set_transient($cache_key, $data, 60);
            return $data;
        }
        
        $result = array(
            'activated' => false,
            'message'   => $data['message'] ?? '验证失败'
        );
        set_transient($cache_key, $result, 60);
        return $result;
    }
    
    /**
     * 提取文章链接的纯文本内容（AJAX）
     */
    public function extract_article_content() {
        check_ajax_referer('acw_rewrite');

        // 文章重写功能需要激活
        $this->ensure_rewrite_activated_or_exit();
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('权限不足');
            return;
        }
        
        $url = isset($_POST['url']) ? esc_url_raw($_POST['url']) : '';
        
        if (empty($url)) {
            wp_send_json_error('请提供有效的URL');
            return;
        }
        
        // 验证URL格式
        if (!filter_var($url, FILTER_VALIDATE_URL)) {
            wp_send_json_error('无效的URL格式');
            return;
        }
        
        // 检测是否是微信公众号链接
        $is_weixin = (strpos($url, 'mp.weixin.qq.com') !== false);
        
        // 获取网页内容
        // 微信公众号需要更真实的浏览器请求头
        if ($is_weixin) {
            $user_agent = 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36';
            $headers = array(
                'Accept' => 'text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/wxpic,image/tpg,image/webp,image/apng,*/*;q=0.8,application/signed-exchange;v=b3;q=0.7',
                'Accept-Language' => 'zh-CN,zh;q=0.9,en;q=0.8',
                'Accept-Encoding' => 'gzip, deflate, br',
                'Connection' => 'keep-alive',
                'Upgrade-Insecure-Requests' => '1',
                'Sec-Fetch-Dest' => 'document',
                'Sec-Fetch-Mode' => 'navigate',
                'Sec-Fetch-Site' => 'none',
                'Sec-Fetch-User' => '?1',
                'Cache-Control' => 'max-age=0',
                'Referer' => 'https://mp.weixin.qq.com/'
            );
        } else {
            $user_agent = 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36';
            $headers = array(
                'Accept' => 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,image/apng,*/*;q=0.8',
                'Accept-Language' => 'zh-CN,zh;q=0.9,en;q=0.8',
                'Accept-Encoding' => 'gzip, deflate, br',
                'Connection' => 'keep-alive',
                'Upgrade-Insecure-Requests' => '1',
                'Sec-Fetch-Dest' => 'document',
                'Sec-Fetch-Mode' => 'navigate',
                'Sec-Fetch-Site' => 'none',
                'Cache-Control' => 'max-age=0'
            );
        }
        
        $response = wp_remote_get($url, array(
            'timeout' => 30,
            'sslverify' => false,
            'user-agent' => $user_agent,
            'headers' => $headers,
            'redirection' => 5,
            'cookies' => array() // 允许Cookie
        ));
        
        if (is_wp_error($response)) {
            wp_send_json_error('无法获取网页内容：' . $response->get_error_message());
            return;
        }
        
        $body = wp_remote_retrieve_body($response);
        $http_code = wp_remote_retrieve_response_code($response);
        
        if ($http_code !== 200) {
            wp_send_json_error('HTTP错误：' . $http_code);
            return;
        }
        
        if (empty($body)) {
            wp_send_json_error('网页内容为空');
            return;
        }
        
        // 检测是否是验证页面（微信公众号反爬虫）
        if (preg_match('/环境异常|完成验证|安全验证|验证码|访问异常/i', $body)) {
            wp_send_json_error('该链接需要验证，无法直接提取。请尝试：1. 在浏览器中打开链接并完成验证；2. 复制页面内容后手动输入；3. 使用其他可访问的文章链接。');
            return;
        }
        
        // 调试：检查是否包含微信公众号文章的关键标识
        $has_js_content = (strpos($body, 'id="js_content"') !== false || strpos($body, 'js_content') !== false);
        $has_rich_media = (strpos($body, 'rich_media_content') !== false);
        
        // 如果检测到是微信公众号但找不到关键标识，可能是内容被JavaScript动态加载
        if ($is_weixin && !$has_js_content && !$has_rich_media) {
            // 尝试检查是否有其他内容标识
            if (strpos($body, 'activity-name') === false && strpos($body, 'rich_media_title') === false) {
                // 可能是JavaScript渲染的内容，尝试提取所有可见文本
            }
        }
        
        // 提取标题
        $title = '';
        if (preg_match('/<title[^>]*>(.*?)<\/title>/is', $body, $matches)) {
            $title = trim(strip_tags($matches[1]));
            // 清理标题中的特殊字符
            $title = preg_replace('/[\s\n\r]+/', ' ', $title);
        }
        
        // 微信公众号文章标题可能在特定位置
        if ($is_weixin && empty($title)) {
            if (preg_match('/<h1[^>]*id=["\']activity-name["\'][^>]*>(.*?)<\/h1>/is', $body, $matches)) {
                $title = trim(strip_tags($matches[1]));
            } elseif (preg_match('/<h1[^>]*class=["\'][^"\']*rich_media_title[^"\']*["\'][^>]*>(.*?)<\/h1>/is', $body, $matches)) {
                $title = trim(strip_tags($matches[1]));
            } elseif (preg_match('/<h1[^>]*>(.*?)<\/h1>/is', $body, $matches)) {
                $title = trim(strip_tags($matches[1]));
            }
        } elseif (empty($title) && preg_match('/<h1[^>]*>(.*?)<\/h1>/is', $body, $matches)) {
            $title = trim(strip_tags($matches[1]));
        }
        
        // 使用DOMDocument提取正文内容
        $content = '';
        
        // 尝试使用DOMDocument
        if (class_exists('DOMDocument')) {
            libxml_use_internal_errors(true);
            $dom = new DOMDocument();
            @$dom->loadHTML(mb_convert_encoding($body, 'HTML-ENTITIES', 'UTF-8'));
            libxml_clear_errors();
            
            $xpath = new DOMXPath($dom);
            
            // 移除不需要的元素
            $remove_tags = array('script', 'style', 'nav', 'header', 'footer', 'aside', 'form', 'iframe', 'noscript');
            foreach ($remove_tags as $tag) {
                $elements = $dom->getElementsByTagName($tag);
                $to_remove = array();
                foreach ($elements as $el) {
                    $to_remove[] = $el;
                }
                foreach ($to_remove as $el) {
                    if ($el->parentNode) {
                        $el->parentNode->removeChild($el);
                    }
                }
            }
            
            // 移除常见的非正文类名元素
            $exclude_classes = array('nav', 'navigation', 'menu', 'sidebar', 'widget', 'ad', 'advertisement', 'comment', 'comments', 'share', 'social', 'related', 'footer', 'header', 'breadcrumb', 'pagination');
            foreach ($exclude_classes as $class) {
                $nodes = $xpath->query("//*[contains(@class, '{$class}')]");
                $to_remove = array();
                foreach ($nodes as $node) {
                    $to_remove[] = $node;
                }
                foreach ($to_remove as $node) {
                    if ($node->parentNode) {
                        $node->parentNode->removeChild($node);
                    }
                }
            }
            
            // 优先查找常见的正文容器
            $content_selectors = array();
            
            // 微信公众号文章的特殊选择器
            if ($is_weixin) {
                $content_selectors = array(
                    '//*[@id="js_content"]',  // 微信公众号文章内容区域（最常用）
                    '//div[@id="js_content"]',
                    '//*[@id="js_content"]//p',  // 直接提取段落
                    '//*[contains(@class, "rich_media_content")]',
                    '//*[contains(@class, "js_content")]',
                    '//div[contains(@class, "rich_media_area")]//div[contains(@class, "rich_media_content")]',
                    '//div[contains(@class, "rich_media_content")]',
                    '//section[contains(@class, "rich_media")]//div[contains(@class, "rich_media_content")]',
                    '//div[contains(@id, "js_content")]//*[self::p or self::span or self::div]',  // 更宽松的选择器
                    '//*[contains(@class, "rich_media")]//*[contains(@class, "content")]'
                );
            }
            
            // 通用选择器
            $content_selectors = array_merge($content_selectors, array(
                '//article',
                '//main',
                '//*[contains(@class, "post-content")]',
                '//*[contains(@class, "entry-content")]',
                '//*[contains(@class, "article-content")]',
                '//*[contains(@class, "content") and not(contains(@class, "nav")) and not(contains(@class, "menu"))]',
                '//*[contains(@id, "content") and not(contains(@id, "nav"))]',
                '//*[contains(@class, "post-body")]',
                '//*[contains(@class, "article-body")]',
                '//*[contains(@class, "text-content")]'
            ));
            
            $best_content = '';
            $max_length = 0;
            
            foreach ($content_selectors as $selector) {
                $nodes = $xpath->query($selector);
                foreach ($nodes as $node) {
                    // 对于微信公众号，使用更宽松的提取策略
                    if ($is_weixin) {
                        // 先尝试提取所有段落
                        $paragraphs = $xpath->query('.//p', $node);
                        $text_content = '';
                        foreach ($paragraphs as $p) {
                            $text = trim($p->textContent);
                            // 微信公众号段落可能较短，降低阈值
                            if (mb_strlen($text) > 10) {
                                $text_content .= $text . "\n\n";
                            }
                        }
                        
                        // 如果段落文本太少，提取所有文本（包括span、div等）
                        if (mb_strlen($text_content) < 200) {
                            // 提取所有文本节点，但排除script、style等
                            $text_content = trim($node->textContent);
                        }
                    } else {
                        // 普通网站：只提取段落文本
                        $paragraphs = $xpath->query('.//p', $node);
                        $text_content = '';
                        foreach ($paragraphs as $p) {
                            $text = trim($p->textContent);
                            if (mb_strlen($text) > 20) {
                                $text_content .= $text . "\n\n";
                            }
                        }
                        
                        // 如果段落文本太少，尝试提取所有文本
                        if (mb_strlen($text_content) < 100) {
                            $text_content = trim($node->textContent);
                        }
                    }
                    
                    // 选择文本最长的内容区域
                    if (mb_strlen($text_content) > $max_length) {
                        $max_length = mb_strlen($text_content);
                        $best_content = $text_content;
                    }
                }
            }
            
            // 如果找到了正文内容
            if (!empty($best_content)) {
                $content = $best_content;
            } else {
                // 如果没找到特定区域，尝试直接提取所有段落
                $paragraphs = $xpath->query('//p');
                $text_content = '';
                foreach ($paragraphs as $p) {
                    $text = trim($p->textContent);
                    // 对于微信公众号，使用更宽松的阈值
                    $min_length = $is_weixin ? 10 : 20;
                    if (mb_strlen($text) > $min_length) {
                        $text_content .= $text . "\n\n";
                    }
                }
                if (!empty($text_content)) {
                    $content = $text_content;
                } elseif ($is_weixin) {
                    // 微信公众号：如果段落提取失败，尝试多种策略
                    // 策略1：尝试查找所有包含文本的div
                    $divs = $xpath->query('//div[contains(@class, "rich_media") or contains(@id, "js")]');
                    $div_texts = array();
                    foreach ($divs as $div) {
                        $div_text = trim($div->textContent);
                        if (mb_strlen($div_text) > 100) {
                            $div_texts[] = $div_text;
                        }
                    }
                    if (!empty($div_texts)) {
                        // 选择最长的div文本
                        usort($div_texts, function($a, $b) {
                            return mb_strlen($b) - mb_strlen($a);
                        });
                        $content = $div_texts[0];
                    } else {
                        // 策略2：提取body中的所有文本（排除已移除的元素）
                        $body_node = $dom->getElementsByTagName('body')->item(0);
                        if ($body_node) {
                            $body_text = trim($body_node->textContent);
                            // 尝试找到最长的连续文本块
                            $lines = explode("\n", $body_text);
                            $longest_lines = array();
                            foreach ($lines as $line) {
                                $line = trim($line);
                                if (mb_strlen($line) > 30) {
                                    $longest_lines[] = $line;
                                }
                            }
                            if (!empty($longest_lines)) {
                                $content = implode("\n\n", $longest_lines);
                            }
                        }
                    }
                }
            }
        } else {
            // 如果DOMDocument不可用，使用简单的strip_tags
            $content = strip_tags($body);
        }
        
        // 如果微信公众号仍然没有提取到内容，尝试使用正则表达式直接提取
        if ($is_weixin && empty($content)) {
            // 尝试从HTML中直接提取js_content区域的内容
            if (preg_match('/<div[^>]*id=["\']js_content["\'][^>]*>(.*?)<\/div>/is', $body, $matches)) {
                $html_content = $matches[1];
                // 移除所有HTML标签，只保留文本
                $html_content = preg_replace('/<script[^>]*>.*?<\/script>/is', '', $html_content);
                $html_content = preg_replace('/<style[^>]*>.*?<\/style>/is', '', $html_content);
                $html_content = strip_tags($html_content);
                $html_content = preg_replace('/\s+/', ' ', $html_content);
                $html_content = trim($html_content);
                if (mb_strlen($html_content) > 100) {
                    $content = $html_content;
                }
            }
            
            // 如果还是失败，尝试提取所有段落标签的内容
            if (empty($content) && preg_match_all('/<p[^>]*>(.*?)<\/p>/is', $body, $p_matches)) {
                $paragraphs = array();
                foreach ($p_matches[1] as $p_html) {
                    $p_text = strip_tags($p_html);
                    $p_text = preg_replace('/\s+/', ' ', $p_text);
                    $p_text = trim($p_text);
                    if (mb_strlen($p_text) > 20) {
                        $paragraphs[] = $p_text;
                    }
                }
                if (!empty($paragraphs)) {
                    $content = implode("\n\n", $paragraphs);
                }
            }
        }
        
        // 清理文本内容
        $content = preg_replace('/\s+/', ' ', $content); // 合并多个空白字符
        $content = preg_replace('/[\r\n]{3,}/', "\n\n", $content); // 规范化换行，最多两个连续换行
        $content = trim($content);
        
        // 检测内容是否主要是JavaScript代码（包含大量JS关键字和符号）
        if (!empty($content)) {
            $js_keywords = array('function', 'var ', 'let ', 'const ', 'return', 'if(', 'else', 'Math.', 'Object.', 'Array.', 'String.', 'Number.', 'Boolean.', 'this.', 'prototype', 'constructor', 'typeof', 'instanceof', 'new ', '=>', '&&', '||', '===', '!==', 'null', 'undefined', 'true', 'false');
            $js_symbols = array('{', '}', '(', ')', '[', ']', ';', '=', '+', '-', '*', '/', '%', '!', '&', '|', '<', '>', '?', ':');
            
            $js_keyword_count = 0;
            $js_symbol_count = 0;
            $total_words = mb_strlen($content);
            
            foreach ($js_keywords as $keyword) {
                $js_keyword_count += substr_count($content, $keyword);
            }
            foreach ($js_symbols as $symbol) {
                $js_symbol_count += substr_count($content, $symbol);
            }
            
            // 如果JavaScript关键字和符号占比过高，可能是JS代码
            $js_ratio = ($js_keyword_count * 10 + $js_symbol_count) / max($total_words, 1);
            if ($js_ratio > 0.3) {
                $content = ''; // 清空内容，重新提取
            }
        }
        
        // 检测是否是视频内容
        $is_video = false;
        // 检查HTML中是否包含视频标签或视频相关的类名
        if (preg_match('/<video[^>]*>/i', $body) || 
            preg_match('/<iframe[^>]*youtube|bilibili|youku|iqiyi|tencent|video/i', $body) ||
            preg_match('/class=["\'][^"\']*video[^"\']*["\']/i', $body) ||
            preg_match('/id=["\'][^"\']*video[^"\']*["\']/i', $body) ||
            preg_match('/视频|video|播放|play/i', $title)) {
            $is_video = true;
        }
        
        // 如果内容为空或主要是JS代码，且检测到视频标识，则认为是视频
        if (empty($content) || (mb_strlen($content) < 50 && $is_video)) {
            $content = '视频';
        }
        
        // 按行分割内容
        $lines = explode("\n", $content);
        $filtered_lines = array();
        $found_first_paragraph = false; // 标记是否找到第一个有意义的段落
        
        foreach ($lines as $line) {
            $line = trim($line);
            if (empty($line)) {
                continue;
            }
            
            // 跳过导航链接（通常是短文本，用空格分隔的多个词，且长度较短）
            if (mb_strlen($line) < 50 && preg_match('/^[\w\s\u4e00-\u9fa5]{1,50}$/u', $line) && 
                preg_match('/\s+/', $line) && mb_strlen($line) < 30) {
                // 可能是导航链接，跳过
                continue;
            }
            
            // 跳过面包屑导航（包含">"或"»"的行）
            if (preg_match('/[>»]/', $line) && mb_strlen($line) < 100) {
                continue;
            }
            
            // 跳过日期格式的行（如"2025-12-26"）
            if (preg_match('/^\d{4}[-年]\d{1,2}[-月]\d{1,2}[日]?/', $line) && mb_strlen($line) < 50) {
                continue;
            }
            
            // 跳过包含常见非正文关键词的行
            $exclude_keywords = '首页|关于|联系我们|登录|注册|搜索|菜单|导航|分享|收藏|点赞|评论|相关文章|热门文章|标签|分类|归档|订阅|RSS|领袖关切|即时中国|国际风云|独家采编|现代西藏|古道新风|籽籽同心|萨班论坛|乡村振兴|本网动态|数据|视频|美图|法治|旅游|文化|生态|教育|藏医药|时尚|宗教|理论|中国西藏网';
            if ($is_weixin) {
                // 微信公众号：添加特定的排除关键词
                $exclude_keywords .= '|视频|小程序|赞|在看|轻点两下|取消赞|取消在看';
            }
            if (preg_match('/^(' . $exclude_keywords . ')/i', $line)) {
                continue;
            }
            
            // 跳过过短的行（可能是导航、按钮等）
            $min_length = $is_weixin ? 15 : 30; // 微信公众号使用更宽松的阈值
            if (mb_strlen($line) < $min_length) {
                continue;
            }
            
            // 找到第一个有意义的段落（通常包含完整句子，长度较长）
            if (!$found_first_paragraph) {
                // 检查是否是正文开始
                // 1. 包含完整句子（有标点符号）
                // 2. 长度较长（大于50字符）
                // 3. 或者以常见的正文开头词开始（据、根据、近日、日前等）
                $is_paragraph_start = false;
                
                // 检查是否以常见正文开头词开始
                if (preg_match('/^(据|根据|近日|日前|日前|日前|日前|新华社|本报|记者|本报讯|中新网|人民网|新华网)/u', $line)) {
                    $is_paragraph_start = true;
                }
                // 检查是否包含完整句子（有中文标点符号）
                elseif (mb_strlen($line) > 50 && preg_match('/[。！？；，]/u', $line)) {
                    // 进一步检查：如果包含多个标点符号，更可能是正文
                    $punctuation_count = preg_match_all('/[。！？；，]/u', $line);
                    if ($punctuation_count >= 1) {
                        $is_paragraph_start = true;
                    }
                }
                // 检查是否是较长的段落（大于100字符）
                elseif (mb_strlen($line) > 100) {
                    $is_paragraph_start = true;
                }
                
                if ($is_paragraph_start) {
                    $found_first_paragraph = true;
                    $filtered_lines[] = $line;
                }
            } else {
                // 已经找到第一个段落，继续添加后续内容
                // 对于微信公众号，使用更宽松的阈值
                $min_length = $is_weixin ? 10 : 20;
                if (mb_strlen($line) > $min_length) {
                    $filtered_lines[] = $line;
                }
            }
        }
        
        // 如果没找到第一个段落，使用所有过滤后的行
        if (!$found_first_paragraph && !empty($filtered_lines)) {
            // 重新处理，找到第一个最长的行作为开始
            $max_len = 0;
            $start_idx = 0;
            foreach ($filtered_lines as $idx => $line) {
                if (mb_strlen($line) > $max_len) {
                    $max_len = mb_strlen($line);
                    $start_idx = $idx;
                }
            }
            // 从最长的行开始提取
            $filtered_lines = array_slice($filtered_lines, $start_idx);
        }
        
        $content = implode("\n\n", $filtered_lines);
        $content = trim($content);
        
        // 检测内容是否主要是JavaScript代码（最终检测）
        if (!empty($content) && $content !== '视频') {
            // 检测是否包含大量JavaScript代码特征
            $js_patterns = array(
                '/function\s*\w+\s*\(/',
                '/var\s+\w+\s*=/',
                '/Math\.\w+/',
                '/Object\.\w+/',
                '/Array\.\w+/',
                '/\.prototype/',
                '/\.constructor/',
                '/typeof\s+\w+/',
                '/instanceof\s+\w+/',
                '/=>\s*\{/',
                '/&&|\|\||===|!==/'
            );
            
            $js_match_count = 0;
            foreach ($js_patterns as $pattern) {
                if (preg_match_all($pattern, $content, $matches)) {
                    $js_match_count += count($matches[0]);
                }
            }
            
            // 如果匹配到多个JS模式，且内容中JS符号过多，认为是JS代码
            if ($js_match_count >= 5) {
                $js_chars = array('{', '}', '(', ')', '[', ']', ';', '=', '&', '|', '!');
                $js_char_count = 0;
                foreach ($js_chars as $char) {
                    $js_char_count += substr_count($content, $char);
                }
                $js_char_ratio = $js_char_count / max(mb_strlen($content), 1);
                
                if ($js_char_ratio > 0.1) {
                    // 可能是JS代码，检查是否有视频标识
                    if ($is_video || preg_match('/视频|video|播放/i', $title) || preg_match('/视频|video|播放/i', $body)) {
                        $content = '视频';
                    } else {
                        $content = ''; // 清空无效内容
                    }
                }
            }
        }
        
        // 如果内容为空或过短，再次检测是否是视频
        if (empty($content) || (mb_strlen($content) < 50 && !preg_match('/[。！？；，]/u', $content))) {
            // 检查是否包含视频相关关键词
            if ($is_video || 
                preg_match('/视频|video|播放|play|观看|点击播放/i', $body) || 
                preg_match('/<video|<iframe[^>]*(youtube|bilibili|youku|iqiyi|tencent|video)/i', $body) ||
                preg_match('/视频|video|播放/i', $title)) {
                $content = '视频';
            }
        }
        
        if (empty($content)) {
            $error_msg = '未能提取到有效内容';
            if ($is_weixin) {
                $error_msg .= '。微信公众号文章可能需要：1. 在浏览器中打开链接并完成验证；2. 文章可能需要登录才能查看；3. 请尝试使用其他可访问的文章链接。';
            }
            wp_send_json_error($error_msg);
            return;
        }
        
        wp_send_json_success(array(
            'title' => $title,
            'content' => $content,
            'url' => $url
        ));
    }
    
    /**
     * 添加文章链接到队列（AJAX）
     */
    public function add_article_to_list() {
        check_ajax_referer('acw_rewrite');

        // 文章重写功能需要激活
        $this->ensure_rewrite_activated_or_exit();
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('权限不足');
            return;
        }
        
        $url = isset($_POST['url']) ? esc_url_raw($_POST['url']) : '';
        $title = isset($_POST['title']) ? sanitize_text_field($_POST['title']) : '';
        
        if (empty($url)) {
            wp_send_json_error('请提供有效的URL');
            return;
        }
        
        // 验证URL格式
        if (!filter_var($url, FILTER_VALIDATE_URL)) {
            wp_send_json_error('无效的URL格式');
            return;
        }
        
        // 如果没有提供标题，使用URL作为标题
        if (empty($title)) {
            $title = $url;
        }
        
        // 获取现有列表
        $article_list = get_option('acw_article_list', array());
        
        // 检查是否已存在
        foreach ($article_list as $item) {
            if ($item['url'] === $url) {
                wp_send_json_error('该链接已存在于列表中');
                return;
            }
        }
        
        // 添加到列表
        $article_list[] = array(
            'url' => $url,
            'title' => $title,
            'added_time' => current_time('Y-m-d H:i:s'),
            'id' => uniqid('acw_', true)
        );
        
        // 保存列表
        update_option('acw_article_list', $article_list);
        
        wp_send_json_success(array(
            'message' => '已添加到列表',
            'list' => $article_list
        ));
    }
    
    /**
     * 从列表移除文章链接（AJAX）
     */
    public function remove_article_from_list() {
        check_ajax_referer('acw_rewrite');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('权限不足');
            return;
        }
        
        $id = isset($_POST['id']) ? sanitize_text_field($_POST['id']) : '';
        
        if (empty($id)) {
            wp_send_json_error('请提供有效的ID');
            return;
        }
        
        // 获取现有列表
        $article_list = get_option('acw_article_list', array());
        
        // 移除指定项
        $article_list = array_filter($article_list, function($item) use ($id) {
            return $item['id'] !== $id;
        });
        
        // 重新索引数组
        $article_list = array_values($article_list);
        
        // 保存列表
        update_option('acw_article_list', $article_list);
        
        wp_send_json_success(array(
            'message' => '已从列表移除',
            'list' => $article_list
        ));
    }
    
    /**
     * 获取文章链接列表（AJAX）
     */
    public function get_article_list() {
        check_ajax_referer('acw_rewrite');

        // 文章重写功能需要激活
        $this->ensure_rewrite_activated_or_exit();
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('权限不足');
            return;
        }
        
        $article_list = get_option('acw_article_list', array());
        
        wp_send_json_success(array(
            'list' => $article_list
        ));
    }
    
    /**
     * 创建重写文章（AJAX）- 仅创建新文章，不执行重写
     */
    public function create_rewrite_post() {
        // 验证nonce
        $nonce = isset($_POST['_ajax_nonce']) ? $_POST['_ajax_nonce'] : (isset($_POST['nonce']) ? $_POST['nonce'] : '');
        if (!wp_verify_nonce($nonce, 'acw_rewrite')) {
            wp_send_json_error('安全验证失败');
            return;
        }

        // 文章重写功能需要激活
        $this->ensure_rewrite_activated_or_exit();
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('权限不足');
            return;
        }
        
        $id = isset($_POST['id']) ? sanitize_text_field($_POST['id']) : '';
        $url = isset($_POST['url']) ? esc_url_raw($_POST['url']) : '';
        $title = isset($_POST['title']) ? sanitize_text_field($_POST['title']) : '';
        
        if (empty($id) || empty($url)) {
            wp_send_json_error('参数不完整');
            return;
        }
        
        // 获取文章列表中的标题作为默认标题
        $article_list = get_option('acw_article_list', array());
        $article_item = null;
        foreach ($article_list as $item) {
            if ($item['id'] === $id) {
                $article_item = $item;
                break;
            }
        }
        
        $post_title = !empty($title) ? $title : (isset($article_item['title']) ? $article_item['title'] : '重写文章');
        
        // 创建新文章（草稿状态）
        $post_data = array(
            'post_title' => $post_title,
            'post_content' => '正在重写中...',
            'post_status' => 'draft',
            'post_type' => 'post',
            'post_author' => get_current_user_id()
        );
        
        $post_id = wp_insert_post($post_data);
        
        if (is_wp_error($post_id)) {
            wp_send_json_error('创建文章失败: ' . $post_id->get_error_message());
            return;
        }
        
        $edit_link = get_edit_post_link($post_id, 'raw');
        
        // 保存 post_id 到文章列表
        foreach ($article_list as &$item) {
            if ($item['id'] === $id) {
                $item['post_id'] = $post_id;
                break;
            }
        }
        update_option('acw_article_list', $article_list);
        
        wp_send_json_success(array(
            'post_id' => $post_id,
            'edit_link' => $edit_link,
            'title' => $post_title
        ));
    }
    
    /**
     * 重写文章（AJAX）
     */
    public function rewrite_article() {
        error_log('=== rewrite_article called ===');
        
        // 验证nonce（支持_ajax_nonce和nonce两种参数名）
        $nonce = isset($_POST['_ajax_nonce']) ? $_POST['_ajax_nonce'] : (isset($_POST['nonce']) ? $_POST['nonce'] : '');
        if (!wp_verify_nonce($nonce, 'acw_rewrite')) {
            error_log('Nonce验证失败');
            wp_send_json_error('安全验证失败');
            return;
        }

        // 文章重写功能需要激活
        $this->ensure_rewrite_activated_or_exit();
        
        if (!current_user_can('manage_options')) {
            error_log('权限不足');
            wp_send_json_error('权限不足');
            return;
        }
        
        $id = isset($_POST['id']) ? sanitize_text_field($_POST['id']) : '';
        $url = isset($_POST['url']) ? esc_url_raw($_POST['url']) : '';
        $title = isset($_POST['title']) ? sanitize_text_field($_POST['title']) : '';
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        
        error_log('重写参数 - ID: ' . $id . ', URL: ' . $url . ', Title: ' . $title . ', Post ID: ' . $post_id);
        
        if (empty($id) || empty($url)) {
            error_log('参数不完整');
            wp_send_json_error('参数不完整');
            return;
        }
        
        // 获取文章列表
        $article_list = get_option('acw_article_list', array());
        $article_item = null;
        foreach ($article_list as $item) {
            if ($item['id'] === $id) {
                $article_item = $item;
                break;
            }
        }
        
        if (!$article_item) {
            error_log('未找到文章项');
            wp_send_json_error('未找到文章项');
            return;
        }
        
        // 获取文章内容和标题
        error_log('开始提取文章内容...');
        $extract_response = wp_remote_get($url, array(
            'timeout' => 30,
            'sslverify' => false,
            'user-agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
            'headers' => array(
                'Accept' => 'text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
                'Accept-Language' => 'zh-CN,zh;q=0.9,en;q=0.8'
            ),
            'redirection' => 5,
            'cookies' => array()
        ));
        
        if (is_wp_error($extract_response)) {
            error_log('提取文章内容失败: ' . $extract_response->get_error_message());
            wp_send_json_error('提取文章内容失败: ' . $extract_response->get_error_message());
            return;
        }
        
        $http_code = wp_remote_retrieve_response_code($extract_response);
        error_log('提取文章HTTP状态码: ' . $http_code);
        if ($http_code !== 200) {
            wp_send_json_error('HTTP错误：' . $http_code);
            return;
        }

        $body = wp_remote_retrieve_body($extract_response);
        if (empty($body)) {
            wp_send_json_error('网页内容为空');
            return;
        }

        // 检测是否是验证/反爬页面
        if (preg_match('/环境异常|完成验证|安全验证|验证码|访问异常/i', $body)) {
            wp_send_json_error('该链接需要验证，无法直接提取。请尝试：1. 在浏览器中打开链接并完成验证；2. 更换可访问的文章链接。');
            return;
        }

        $is_weixin = (strpos($url, 'mp.weixin.qq.com') !== false);
        // 优先使用文章列表中的标题，然后尝试从HTML中提取
        $original_title = isset($article_item['title']) ? $article_item['title'] : $title;
        $original_content = '';
        
        // 提取标题（如果从HTML中提取到标题，则使用提取的标题）
        if (preg_match('/<title[^>]*>(.*?)<\/title>/is', $body, $matches)) {
            $extracted_title = trim(strip_tags($matches[1]));
            if (!empty($extracted_title)) {
                $original_title = $extracted_title;
            }
        } elseif (preg_match('/<h1[^>]*>(.*?)<\/h1>/is', $body, $matches)) {
            $extracted_title = trim(strip_tags($matches[1]));
            if (!empty($extracted_title)) {
                $original_title = $extracted_title;
            }
        }
        
        // 如果标题仍然为空，使用文章列表中的标题或POST参数中的标题作为备用
        if (empty($original_title)) {
            $original_title = isset($article_item['title']) ? $article_item['title'] : $title;
        }
        
        // 提取内容（使用与 extract_article_content 类似的“候选节点打分”策略，兼容更多站点模板）
        if (class_exists('DOMDocument')) {
            libxml_use_internal_errors(true);
            $dom = new DOMDocument();
            @$dom->loadHTML(mb_convert_encoding($body, 'HTML-ENTITIES', 'UTF-8'));
            libxml_clear_errors();
            
            $xpath = new DOMXPath($dom);
            
            // 移除不需要的元素
            $remove_tags = array('script', 'style', 'nav', 'header', 'footer', 'aside', 'form', 'iframe', 'noscript');
            foreach ($remove_tags as $tag) {
                $elements = $dom->getElementsByTagName($tag);
                $to_remove = array();
                foreach ($elements as $el) {
                    $to_remove[] = $el;
                }
                foreach ($to_remove as $el) {
                    if ($el->parentNode) {
                        $el->parentNode->removeChild($el);
                    }
                }
            }

            // 移除常见的非正文类名元素（降低噪音）
            $exclude_classes = array('nav', 'navigation', 'menu', 'sidebar', 'widget', 'ad', 'advertisement', 'comment', 'comments', 'share', 'social', 'related', 'footer', 'header', 'breadcrumb', 'pagination', 'recommend');
            foreach ($exclude_classes as $class) {
                $nodes = $xpath->query("//*[contains(@class, '{$class}')]");
                $to_remove = array();
                foreach ($nodes as $node) {
                    $to_remove[] = $node;
                }
                foreach ($to_remove as $node) {
                    if ($node->parentNode) {
                        $node->parentNode->removeChild($node);
                    }
                }
            }
            
            $content_selectors = array(
                // 微信公众号
                '//*[@id="js_content"]',
                '//*[contains(@class, "rich_media_content")]',
                // 通用
                '//article',
                '//main',
                '//*[contains(@class, "post-content")]',
                '//*[contains(@class, "entry-content")]',
                '//*[contains(@class, "article-content")]',
                '//*[contains(@class, "article_body")]',
                '//*[contains(@class, "article-body")]',
                '//*[contains(@class, "post-body")]',
                '//*[contains(@class, "text-content")]',
                // 容器兜底：content 但排除导航/菜单
                '//*[contains(@class, "content") and not(contains(@class, "nav")) and not(contains(@class, "menu"))]',
                '//*[contains(@id, "content") and not(contains(@id, "nav"))]'
            );

            $best_content = '';
            $max_length = 0;
            foreach ($content_selectors as $selector) {
                $nodes = $xpath->query($selector);
                foreach ($nodes as $node) {
                    $paragraphs = $xpath->query('.//p', $node);
                    $text_content = '';
                    foreach ($paragraphs as $p) {
                        $text = trim($p->textContent);
                        $min_len = $is_weixin ? 10 : 20;
                        if (mb_strlen($text) > $min_len) {
                            $text_content .= $text . "\n\n";
                        }
                    }
                    if (mb_strlen($text_content) < ($is_weixin ? 200 : 100)) {
                        $text_content = trim($node->textContent);
                    }
                    if (mb_strlen($text_content) > $max_length) {
                        $max_length = mb_strlen($text_content);
                        $best_content = $text_content;
                    }
                }
            }

            if (!empty($best_content)) {
                $original_content = trim($best_content);
            } else {
                // 兜底：提取全局段落
                $paragraphs = $xpath->query('//p');
                $text_content = '';
                foreach ($paragraphs as $p) {
                    $text = trim($p->textContent);
                    $min_len = $is_weixin ? 10 : 20;
                    if (mb_strlen($text) > $min_len) {
                        $text_content .= $text . "\n\n";
                    }
                }
                $original_content = trim($text_content);
            }
        }

        // 清理文本内容（保持换行，用于提示词结构）
        if (!empty($original_content)) {
            $original_content = preg_replace('/[ \t]+/', ' ', $original_content);
            $original_content = preg_replace('/\r\n|\r/', "\n", $original_content);
            $original_content = preg_replace("/\n{3,}/", "\n\n", $original_content);
            $original_content = trim($original_content);
        }
        
        if (empty($original_content)) {
            error_log('未能提取到文章内容');
            wp_send_json_error('未能提取到文章内容');
            return;
        }
        
        error_log('文章内容提取成功，长度: ' . mb_strlen($original_content));
        
        // 获取重写设置
        $use_original_title = get_option('rewrite_use_original_title', 0);
        $rewrite_system_prompt = get_option('rewrite_system_prompt', '');
        $rewrite_prompt_template = get_option('rewrite_prompt_template', '');
        $rewrite_title_prompt_template = get_option('rewrite_title_prompt_template', '');
        
        error_log('使用原标题: ' . ($use_original_title ? '是' : '否'));
        
        // 替换占位符
        $rewrite_prompt = str_replace(array('[原文]', '[标题]'), array($original_content, $original_title), $rewrite_prompt_template);
        
        // 使用 AI通道（多API）配置重写内容，支持失败自动切换（图片生成设置不受影响）
        $apis = get_option('ai_apis', array());
        if (!is_array($apis)) {
            $apis = array();
        }
        if (empty($apis)) {
            error_log('[文章重写] - 未找到AI通道配置');
            wp_send_json_error('请先在 AI通道 中配置至少一个通道');
            return;
        }

        $default_api_id     = get_option('ai_default_api', 'deepseek');
        $deepseek_default_url = 'https://api.deepseek.com/v1/chat/completions';

        if (!is_string($default_api_id) || !isset($apis[$default_api_id])) {
            $api_keys = array_keys($apis);
            $default_api_id = isset($api_keys[0]) ? $api_keys[0] : 'deepseek';
        }

        $api_order = array($default_api_id);
        foreach ($apis as $api_id => $api) {
            if ($api_id !== $default_api_id) {
                $api_order[] = $api_id;
            }
        }

        // 重写文章内容
        error_log('开始重写文章内容...');
        $rewrite_request = array(
            'model' => '',
            'messages' => array(
                array('role' => 'system', 'content' => $rewrite_system_prompt),
                array('role' => 'user', 'content' => $rewrite_prompt)
            ),
            'temperature' => 0.7
        );

        $rewrite_errors = array();
        $rewritten_content = '';
        $used_api_id = $default_api_id;

        foreach ($api_order as $api_id) {
            if (!isset($apis[$api_id]) || !is_array($apis[$api_id])) {
                continue;
            }

            $api = $apis[$api_id];

            // DeepSeek：URL 可更改（为空时使用默认值），模型可默认
            if ($api_id === 'deepseek') {
                $api['name']  = isset($api['name']) && $api['name'] ? $api['name'] : 'DeepSeek';
                if (!isset($api['url']) || trim((string)$api['url']) === '') {
                    $api['url'] = $deepseek_default_url;
                }
                $api['model'] = isset($api['model']) && trim((string)$api['model']) !== '' ? $api['model'] : 'deepseek-chat';
            }

            $api_name = isset($api['name']) && $api['name'] ? $api['name'] : $api_id;
            $missing = array();
            if (empty($api['key']))   $missing[] = 'API Key';
            if (empty($api['url']))   $missing[] = 'URL';
            if (empty($api['model'])) $missing[] = '模型';
            if (!empty($missing)) {
                $rewrite_errors[] = "API {$api_name}: 配置不完整（缺少 " . implode('、', $missing) . "）";
                continue;
            }

            $rewrite_request['model'] = $api['model'];

            $rewrite_response = wp_remote_post($api['url'], array(
                'headers' => array(
                    'Authorization' => 'Bearer ' . $api['key'],
                    'Content-Type' => 'application/json'
                ),
                'body' => json_encode($rewrite_request, JSON_UNESCAPED_UNICODE),
                'timeout' => 300,
                'sslverify' => false
            ));

            if (is_wp_error($rewrite_response)) {
                $rewrite_errors[] = "API {$api_name}: " . $rewrite_response->get_error_message();
                continue;
            }

            $response_code = wp_remote_retrieve_response_code($rewrite_response);
            $response_body = wp_remote_retrieve_body($rewrite_response);

            if ($response_code !== 200) {
                $error_message = json_decode($response_body, true);
                $error_detail  = isset($error_message['error']['message']) ? $error_message['error']['message'] : $response_body;
                $rewrite_errors[] = "API {$api_name}: API响应错误: " . $error_detail;
                continue;
            }

            $rewrite_body = json_decode($response_body, true);
            if (!isset($rewrite_body['choices'][0]['message']['content'])) {
                $rewrite_errors[] = "API {$api_name}: 重写响应格式错误";
                continue;
            }

            $rewritten_content = trim($rewrite_body['choices'][0]['message']['content']);
            if ($rewritten_content === '') {
                $rewrite_errors[] = "API {$api_name}: 重写结果为空";
                continue;
            }

            $used_api_id = $api_id;
            break;
        }

        if ($rewritten_content === '') {
            error_log('[文章重写] - 所有AI通道重写都失败：' . implode(' | ', $rewrite_errors));
            wp_send_json_error("所有API都失败了：\n" . implode("\n", $rewrite_errors));
            return;
        }

        // 成功：若发生切换，则更新默认 API（与标题/文章生成一致）
        if ($used_api_id !== $default_api_id) {
            update_option('ai_default_api', $used_api_id);
        }

        error_log('[文章重写] 步骤4: 文章内容重写成功，长度: ' . mb_strlen($rewritten_content));
        
        // 确定最终标题
        $final_title = '';
        if ($use_original_title) {
            // 情况A：使用原标题
            error_log('[文章重写] 步骤5: 使用原标题模式');
            $final_title = $original_title;
            error_log('[文章重写] - 使用原标题: ' . $final_title);
            
            // 如果标题为空，返回错误
            if (empty($final_title)) {
                error_log('[文章重写] - 原标题为空，无法创建文章');
                wp_send_json_error('无法获取原标题，请检查文章链接或手动输入标题');
                return;
            }
        } else {
            // 情况B：生成新标题
            error_log('[文章重写] 步骤5: 生成新标题模式');
            error_log('[文章重写] - 开始根据重写后的内容生成标题...');
            
            // 使用多API标题生成配置
            $apis = get_option('ai_apis', array());
            if (!is_array($apis)) {
                $apis = array();
            }
            $default_api_id = get_option('ai_default_api', 'deepseek');
            $deepseek_default_url = 'https://api.deepseek.com/v1/chat/completions';
            
            if (empty($apis)) {
                error_log('[文章重写] - 未找到标题生成API配置');
                wp_send_json_error('未找到标题生成API配置');
                return;
            }
            
            // 默认 API 不存在时，回落到第一个可用 API（兼容旧配置 api1）
            if (!is_string($default_api_id) || !isset($apis[$default_api_id])) {
                $api_keys = array_keys($apis);
                $default_api_id = isset($api_keys[0]) ? $api_keys[0] : 'deepseek';
            }

            $api_order = array($default_api_id);
            foreach ($apis as $api_id => $api) {
                if ($api_id !== $default_api_id) {
                    $api_order[] = $api_id;
                }
            }
            
            error_log('[文章重写] - API顺序: ' . implode(', ', $api_order));
            
            // 使用重写标题专用的系统提示词
            $rewrite_title_system_prompt = get_option('rewrite_title_system_prompt', '');
            $title_system_prompt = $rewrite_title_system_prompt;
            
            // 如果重写标题系统提示词为空，使用默认的标题生成系统提示词作为后备
            if (empty($title_system_prompt)) {
                $title_system_prompt = get_option('ai_title_prompt', '');
            }
            
            // 替换占位符：[内容] 和 [标题]
            $title_prompt = str_replace(array('[内容]', '[标题]'), array($rewritten_content, $original_title), $rewrite_title_prompt_template);
            
            error_log('[文章重写] - 标题生成提示词模板: ' . substr($rewrite_title_prompt_template, 0, 100) . '...');
            error_log('[文章重写] - 标题生成提示词（替换后）: ' . substr($title_prompt, 0, 100) . '...');
            
            $title_generated = false;
            foreach ($api_order as $api_id) {
                if (!isset($apis[$api_id])) {
                    continue;
                }
                
                $api = $apis[$api_id];
                
                // DeepSeek：URL 可更改（为空时使用默认值），模型可默认
                if ($api_id === 'deepseek') {
                    $api['name']  = isset($api['name']) && $api['name'] ? $api['name'] : 'DeepSeek';
                    if (!isset($api['url']) || trim((string)$api['url']) === '') {
                        $api['url'] = $deepseek_default_url;
                    }
                    $api['model'] = isset($api['model']) && trim((string)$api['model']) !== '' ? $api['model'] : 'deepseek-chat';
                }

                $api_name = isset($api['name']) && $api['name'] ? $api['name'] : $api_id;
                $missing = array();
                if (empty($api['key']))   $missing[] = 'API Key';
                if (empty($api['url']))   $missing[] = 'URL';
                if (empty($api['model'])) $missing[] = '模型';

                if (!empty($missing)) {
                    error_log('[文章重写] - API ' . $api_name . ' 配置不完整（缺少 ' . implode('、', $missing) . '），跳过');
                    continue;
                }
                
                error_log('[文章重写] - 尝试使用API: ' . $api_name . ' (' . $api_id . ')');
                
                $title_response = wp_remote_post($api['url'], array(
                    'headers' => array(
                        'Authorization' => 'Bearer ' . $api['key'],
                        'Content-Type' => 'application/json',
                    ),
                    'body' => json_encode(array(
                        'model' => $api['model'],
                        'messages' => array(
                            array('role' => 'system', 'content' => $title_system_prompt),
                            array('role' => 'user', 'content' => $title_prompt)
                        ),
                        'temperature' => 1.2
                    )),
                    'timeout' => 300
                ));
                
                if (is_wp_error($title_response)) {
                    error_log('[文章重写] - API ' . $api_name . ' 请求失败: ' . $title_response->get_error_message());
                    continue;
                }
                
                $title_body = json_decode(wp_remote_retrieve_body($title_response), true);
                if (isset($title_body['error']) || !isset($title_body['choices'][0]['message']['content'])) {
                    error_log('[文章重写] - API ' . $api_name . ' 响应格式错误');
                    continue;
                }
                
                $title_content = trim($title_body['choices'][0]['message']['content']);
                $titles = array_filter(array_map('trim', explode("\n", $title_content)));
                
                error_log('[文章重写] - API ' . $api_name . ' 返回标题数量: ' . count($titles));
                
                if (!empty($titles)) {
                    $final_title = trim($titles[0]);
                    $final_title = preg_replace('/^\d+\.\s*/', '', $final_title);
                    $final_title = str_replace(array('！', '《', '》', '[', ']', ':', '：', '**'), '', $final_title);
                    $final_title = trim($final_title, '""\""');
                    $final_title = trim($final_title);
                    
                    if (!empty($final_title)) {
                        error_log('[文章重写] - 标题生成成功: ' . $final_title);
                        $title_generated = true;
                        break;
                    }
                }
            }
            
            if (!$title_generated) {
                error_log('[文章重写] - 所有API都失败，使用原标题作为备用');
                $final_title = $original_title;
            }
        }
        
        // 更新或创建文章
        if ($post_id > 0) {
            // 更新现有文章
            error_log('[文章重写] 步骤6: 开始更新文章');
            error_log('[文章重写] - 文章ID: ' . $post_id);
            error_log('[文章重写] - 文章标题: ' . $final_title);
            error_log('[文章重写] - 文章内容长度: ' . mb_strlen($rewritten_content));
            
            $post_data = array(
                'ID' => $post_id,
                'post_title' => $final_title,
                'post_content' => $rewritten_content
            );
            
            $updated_post_id = wp_update_post($post_data);
            
            if (is_wp_error($updated_post_id)) {
                error_log('[文章重写] - 更新文章失败: ' . $updated_post_id->get_error_message());
                wp_send_json_error('更新文章失败: ' . $updated_post_id->get_error_message());
                return;
            }
            
            error_log('[文章重写] 步骤7: 文章更新成功');
            error_log('[文章重写] - 文章ID: ' . $updated_post_id);
        } else {
            // 创建新文章
            error_log('[文章重写] 步骤6: 开始创建新文章');
            error_log('[文章重写] - 文章标题: ' . $final_title);
            error_log('[文章重写] - 文章内容长度: ' . mb_strlen($rewritten_content));
            
            $post_data = array(
                'post_title' => $final_title,
                'post_content' => $rewritten_content,
                'post_status' => 'draft',
                'post_type' => 'post',
                'post_author' => get_current_user_id()
            );
            
            $post_id = wp_insert_post($post_data);
            
            if (is_wp_error($post_id)) {
                error_log('[文章重写] - 创建文章失败: ' . $post_id->get_error_message());
                wp_send_json_error('创建文章失败: ' . $post_id->get_error_message());
                return;
            }
            
            error_log('[文章重写] 步骤7: 文章创建成功');
            error_log('[文章重写] - 新文章ID: ' . $post_id);
        }
        
        error_log('[文章重写] ========== 重写流程完成 ==========');
        
        $edit_link = get_edit_post_link($post_id, 'raw');
        
        wp_send_json_success(array(
            'post_id' => $post_id,
            'edit_link' => $edit_link,
            'title' => $final_title,
            'content' => $rewritten_content,  // 返回重写后的内容
            'message' => '重写成功'
        ));
    }
    
    /**
     * 获取文章内容（AJAX）
     */
    public function get_post_content() {
        // 验证nonce
        $nonce = isset($_POST['_ajax_nonce']) ? $_POST['_ajax_nonce'] : (isset($_POST['nonce']) ? $_POST['nonce'] : '');
        if (!wp_verify_nonce($nonce, 'acw_rewrite')) {
            wp_send_json_error('安全验证失败');
            return;
        }
        
        if (!current_user_can('edit_posts')) {
            wp_send_json_error('权限不足');
            return;
        }
        
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        
        if ($post_id <= 0) {
            wp_send_json_error('文章ID无效');
            return;
        }
        
        $post = get_post($post_id);
        
        if (!$post) {
            wp_send_json_error('文章不存在');
            return;
        }
        
        wp_send_json_success(array(
            'content' => $post->post_content
        ));
    }
    
    /**
     * 获取未重写的文章列表（AJAX）
     */
    public function get_unrewritten_articles() {
        // 验证nonce
        $nonce = isset($_POST['_ajax_nonce']) ? $_POST['_ajax_nonce'] : (isset($_POST['nonce']) ? $_POST['nonce'] : '');
        if (!wp_verify_nonce($nonce, 'acw_rewrite')) {
            wp_send_json_error('安全验证失败');
            return;
        }

        // 文章重写功能需要激活
        $this->ensure_rewrite_activated_or_exit();
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('权限不足');
            return;
        }
        
        $article_list = get_option('acw_article_list', array());
        
        // 筛选出未重写的文章（status 为 pending 或未设置），跳过重写失败的文章
        $unrewritten = array();
        foreach ($article_list as $item) {
            $status = isset($item['status']) ? $item['status'] : 'pending';
            // 只处理待重写的文章，跳过已成功和已失败的文章
            if ($status === 'pending') {
                $unrewritten[] = $item;
            }
        }
        
        wp_send_json_success(array(
            'list' => $unrewritten
        ));
    }
    
    /**
     * 更新文章状态（AJAX）
     */
    public function update_article_status() {
        // 验证nonce
        $nonce = isset($_POST['_ajax_nonce']) ? $_POST['_ajax_nonce'] : (isset($_POST['nonce']) ? $_POST['nonce'] : '');
        if (!wp_verify_nonce($nonce, 'acw_rewrite')) {
            wp_send_json_error('安全验证失败');
            return;
        }

        // 文章重写功能需要激活
        $this->ensure_rewrite_activated_or_exit();
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('权限不足');
            return;
        }
        
        $id = isset($_POST['id']) ? sanitize_text_field($_POST['id']) : '';
        $status = isset($_POST['status']) ? sanitize_text_field($_POST['status']) : '';
        
        if (empty($id) || empty($status)) {
            wp_send_json_error('参数不完整');
            return;
        }
        
        $article_list = get_option('acw_article_list', array());
        
        foreach ($article_list as &$item) {
            if ($item['id'] === $id) {
                $item['status'] = $status;
                break;
            }
        }
        
        update_option('acw_article_list', $article_list);
        
        wp_send_json_success(array(
            'message' => '状态已更新'
        ));
    }
    
    /**
     * 检查文章状态（AJAX）
     */
    public function check_post_status() {
        // 验证nonce
        $nonce = isset($_POST['_ajax_nonce']) ? $_POST['_ajax_nonce'] : (isset($_POST['nonce']) ? $_POST['nonce'] : '');
        if (!wp_verify_nonce($nonce, 'acw_rewrite')) {
            wp_send_json_error('安全验证失败');
            return;
        }

        // 文章重写功能需要激活
        $this->ensure_rewrite_activated_or_exit();
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('权限不足');
            return;
        }
        
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        
        if (empty($post_id)) {
            wp_send_json_error('参数不完整');
            return;
        }
        
        $post = get_post($post_id);
        
        if (!$post) {
            wp_send_json_error('文章不存在');
            return;
        }
        
        $is_published = ($post->post_status === 'publish');
        
        wp_send_json_success(array(
            'post_id' => $post_id,
            'post_status' => $post->post_status,
            'is_published' => $is_published
        ));
    }
    
    /**
     * 清空文章列表（AJAX）
     */
    public function clear_article_list() {
        // 验证nonce
        $nonce = isset($_POST['_ajax_nonce']) ? $_POST['_ajax_nonce'] : (isset($_POST['nonce']) ? $_POST['nonce'] : '');
        if (!wp_verify_nonce($nonce, 'acw_rewrite')) {
            wp_send_json_error('安全验证失败');
            return;
        }

        // 文章重写功能需要激活
        $this->ensure_rewrite_activated_or_exit();
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('权限不足');
            return;
        }
        
        // 清空文章列表
        update_option('acw_article_list', array());
        
        wp_send_json_success(array(
            'message' => '列表已清空'
        ));
    }
    
    /**
     * 发送重写失败通知邮件（AJAX）
     */
    public function send_rewrite_error_notification() {
        check_ajax_referer('ai_content_writer');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('权限不足');
            return;
        }
        
        $article_id = isset($_POST['article_id']) ? sanitize_text_field($_POST['article_id']) : '';
        $article_title = isset($_POST['article_title']) ? sanitize_text_field($_POST['article_title']) : '';
        $article_url = isset($_POST['article_url']) ? esc_url_raw($_POST['article_url']) : '';
        $error = isset($_POST['error']) ? sanitize_text_field($_POST['error']) : '';
        
        $notification_email = get_option('notification_email');
        $smtp_username = get_option('smtp_username');
        
        if (empty($notification_email) || empty($smtp_username)) {
            wp_send_json_error('未配置通知邮箱');
            return;
        }
        
        $email_subject = '文章重写失败通知';
        $email_message = '<div style="font-family: Arial, sans-serif; padding: 20px; max-width: 600px; margin: 0 auto; line-height: 1.6;">';
        $email_message .= '<h2 style="color: #333; margin-bottom: 20px;">文章重写失败通知</h2>';
        $email_message .= '<p style="margin-bottom: 15px;"><strong>网站地址：</strong>' . get_site_url() . '</p>';
        $email_message .= '<p style="margin-bottom: 15px;"><strong>文章标题：</strong>' . esc_html($article_title) . '</p>';
        $email_message .= '<p style="margin-bottom: 15px;"><strong>文章链接：</strong><a href="' . esc_url($article_url) . '">' . esc_html($article_url) . '</a></p>';
        $email_message .= '<div style="background-color: #fff1f0; padding: 15px; border-left: 4px solid #d63638; margin-top: 15px;">';
        $email_message .= '<strong>错误信息：</strong><br>' . esc_html($error);
        $email_message .= '</div>';
        $email_message .= '</div>';
        
        $headers = array(
            'Content-Type: text/html; charset=UTF-8',
            'From: AI-Content Writer <' . $smtp_username . '>'
        );
        
        $result = wp_mail(
            array($notification_email),
            $email_subject,
            $email_message,
            $headers
        );
        
        if ($result) {
            wp_send_json_success(array('message' => '通知邮件已发送'));
        } else {
            wp_send_json_error('发送邮件失败');
        }
    }
    
    public function check_card_activation() {
        check_ajax_referer('ai_content_writer');
        
        $activation_status = $this->check_remote_activation();
        wp_send_json_success($activation_status);
    }
    
    public function activate_card_frontend() {
        check_ajax_referer('ai_content_writer');
        
        $card_code = sanitize_text_field($_POST['card_code']);
        if (empty($card_code)) {
            wp_send_json_error(array('message' => '请输入卡密代码'));
        }
        
        $domain = home_url();
        
        $response = wp_remote_post($this->card_api_base_url . '/activate.php', array(
            'body'    => json_encode(array(
                'card_code'         => $card_code,
                'domain'            => $domain,
                'plugin_identifier' => $this->plugin_slug
            )),
            'headers' => array(
                'Content-Type' => 'application/json'
            ),
            'timeout' => 15
        ));
        
        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => '网络连接失败，请检查网络连接'));
            return;
        }
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($data['success'])) {
            if ($data['success']) {
                delete_transient('ai_content_writer_activation_cache');
                delete_transient('ai_content_writer_activation_status_cache');
                wp_send_json_success($data);
            } else {
                wp_send_json_error($data);
            }
        } else {
            wp_send_json_error(array('message' => '激活服务器响应异常'));
        }
    }
    
}

new AI_Content_Writer();


